/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest.impl;

import java.math.BigInteger;
import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;

import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;

import org.ow2.jasmine.probe.api.generated.*;
import org.ow2.jasmine.probe.rest.Indicators;

public class JIndicators implements Indicators {

    private JasmineProbeManager jpManager = null;

    public JIndicators(JasmineProbeManager jpManager) {
        this.jpManager = jpManager;
    }
    /**
     * Process HTTP GET requests.
     * Possible URIs are:
     *  /indicator that corresponds to 'indicator-list' command
     * or
     * /indicator?type={typeName} that corresponds to 'indicator-list -t typeName' command
     * or
     * /indicator?verbose="true" that corresponds to 'indicator-list -v' command
     * @return response containing the existent indicators
     */
    @Override
    public Response getIndicators(String type, boolean verbose, UriInfo ui) {
        List<JAXBElement<IndicatorResource>> elementsList = new ArrayList<JAXBElement<IndicatorResource>>();

        List<JasmineIndicator> indicList = jpManager.getIndicators();
        for (JasmineIndicator indic : indicList) {
            String indicType = indic.getType();
            if (type == null || type.equals(indicType)) {
                String indicName = indic.getName();
                IndicatorResource indicElement = new IndicatorResource();
                indicElement.setName(indic.getName());

                URI indicUri = ui.getAbsolutePathBuilder().path(indicName).build();
                indicElement.setHref(indicUri.toString());

                if (verbose) {
                    indicElement.setType(indicType);
                }

                QName name  = new QName("org.ow2.jasmine.probe:probe-config", "indicator");
                JAXBElement xmlElement =  new JAXBElement<IndicatorResource>(name, IndicatorResource.class, indicElement);
                elementsList.add(xmlElement);
            }
        }

        return Response
                        .status(Response.Status.OK)
                        .entity(new GenericEntity<List<JAXBElement<IndicatorResource>>>(elementsList) {})
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    /**
     * Process HTTP GET requests having /indicator/{name} URI
     * Correspond to indicator-list -n name
     * @param name indicator name
     * @param type
     * @return response
     */
    @Override
    public Response getIndicator(String name, UriInfo ui) {

        UriBuilder ub = ui.getAbsolutePathBuilder();
        String indicatorUri = ub.build("").toString();
        String baseUri = indicatorUri.substring(0, indicatorUri.lastIndexOf(name));

        JasmineIndicator jindic;
        try {
            jindic = jpManager.getIndicator(name);
        } catch (JasmineProbeException e) {
            ErrorType err = new ErrorType();
            err.setMessage(e.getMessage());
            QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "error");
            JAXBElement<ErrorType> errElem = new JAXBElement<ErrorType>(qname, ErrorType.class, err);
            return Response
            .status(Response.Status.NOT_FOUND)
            .entity(errElem)
            .type(MediaType.APPLICATION_XML)
            .build();
        }

        IndicatorResource indicator = new IndicatorResource();
        indicator.setName(name);
        indicator.setType(jindic.getType());
        indicator.setHref(indicatorUri);
        int scale = jindic.getScale();
        indicator.setScale(Integer.valueOf(scale));

        List<Link> links = new ArrayList<Link>();
        List<String> sources = jindic.getSources();
        for (String source : sources) {
            Link link = new Link();
            link.setRel("source");
            UriBuilder builder = UriBuilder.fromPath(baseUri);
            URI uri = builder.path(source).build();
            link.setHref(uri.toString());
            links.add(link);
        }
        indicator.setSource(sources);
        indicator.setLink(links);

        List<PropType> property = new ArrayList<PropType>();
        for (String propKey : jindic.getProperties().keySet()) {
            String propValue = jindic.getProperties().get(propKey);
            PropType prop = new PropType();
            prop.setValue(propValue);
            prop.setKey(propKey);
            property.add(prop);
        }
        indicator.setProperty(property);

        QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "indicator");
        JAXBElement<IndicatorResource> indicResource = new JAXBElement<IndicatorResource>(qname, IndicatorResource.class, indicator);

        return Response
                        .status(Response.Status.OK)
                        .entity(indicResource)
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    /**
     * POST http://host:port/jprobe/indicator
     * Create a new indicator
     */
    public Response createIndicator(IndicatorBaseType indicatorDef) {
        JasmineIndicator indicator = new JasmineIndicator();
        indicator.setName(indicatorDef.getName());
        indicator.setScale(indicatorDef.getScale());
        indicator.setType(indicatorDef.getType());

        Map<String, String> properties = new HashMap<String, String>();

        for (PropType prop : indicatorDef.getProperty()) {
            properties.put(prop.getKey(), prop.getValue());
        }
        indicator.setProperties(properties);

        indicator.setSources(indicatorDef.getSource());

        try {
            jpManager.createIndicator(indicator);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();
    }

    @Override
    /**
     * PUT http://host:port/jprobe/indicator/{name}
     * Change an indicator
     * @param indicator name
     * @param indicatorDef updated definition
     */
    public Response changeIndicator(IndicatorBaseType indicatorDef, String name) {
        try {
            jpManager.getIndicator(name);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        JasmineIndicator indicator = new JasmineIndicator();
        indicator.setName(name);
        indicator.setScale(indicatorDef.getScale());
        indicator.setType(indicatorDef.getType());

        Map<String, String> properties = new HashMap<String, String>();

        for (PropType prop : indicatorDef.getProperty()) {
            properties.put(prop.getKey(), prop.getValue());
        }
        indicator.setProperties(properties);

        indicator.setSources(indicatorDef.getSource());

        try {
            jpManager.changeIndicator(indicator);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();

    }
    @Override
    public Response deleteIndicator(String name) {
        try {
            jpManager.removeIndicator(name);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }
        return Response.status(Response.Status.OK).build();
    }

}
