/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.itests.rest;

import java.io.File;
import java.net.HttpURLConnection;
import java.util.List;

import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.WebResource;

import org.ow2.jasmine.probe.api.generated.JsrType;
import org.ow2.jasmine.probe.api.generated.TypeNameType;

import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

public class ChangeTest extends AbsRestTest {

    /**
     * Setup method.
     * - Creation of a data task for the 'pChange' probe
     */
    @BeforeClass
    public void createDataTask() {
        WebResource webResource;
        ClientResponse response;
        
        // Create a data task for the 'pChange' probe
        String sUri = urlBase + "/probe/pChange/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when creating the data task");
    }

    /**
     * Setdown method.
     * - Delete the data task of the 'pChange' probe
     */
    @AfterClass
    public void deleteDataTask() {
        WebResource webResource;
        ClientResponse response;
        
        // Delete the data task of the 'pChange' probe
        String sUri = urlBase + "/probe/pChange/data";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.delete(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when deleting the data task");
    }

    /**
     * Change the path of the 'otst_change' output file of the 'pChange' probe
     * - previous path: otst_change.csv
     * - new path: otst_change_BIS.csv
     * and check no more collects are stored in the old file, but in the new file.
     */
    @Test
    public void changeOutputFile() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;
        long lengthOfTheFirstOutput;
        
        // Stop the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/stop";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when stoping the probe");
        
        // Get the length of the previous output file of the 'pChange' probe
        String fNameOutput = tmpDirPath + "otst_change.csv";
        File fOutput = new File(fNameOutput);
        lengthOfTheFirstOutput = fOutput.length();
        
        // Change the path the output file
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><output xmlns=\"org.ow2.jasmine.probe:probe-config\" name=\"otst_change\" type=\"file\"><property key=\"path\" value=\""+tmpDirPath+"otst_change_BIS.csv\"/></output>";
        sUri = urlBase + "/output/otst_change";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when changing the output path");

        // Restart the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/start";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when starting the probe");
        
        // Check collects are stored the new output file and no more in the previous old file
        logger.info("Wait a while to get time to do some collects ...");
        Thread.sleep(14*1000);
        fNameOutput = tmpDirPath + "otst_change.csv";
        fOutput = new File(fNameOutput);
        Assert.assertEquals(lengthOfTheFirstOutput, fOutput.length(), "Old output file with path 'otst_change.csv' is modified");
        fNameOutput = tmpDirPath + "otst_change_BIS.csv";
        fOutput = new File(fNameOutput);
        Assert.assertTrue(fOutput.length() > 0, "New output file with path 'otst_change_BIS.csv' is empty");
        
    }

    /**
     * Change an unknown output.
     * We must got an error status.
     */
    @Test
    public void changeUnknownOutput() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;
        
        // Change the unknown output
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><output xmlns=\"org.ow2.jasmine.probe:probe-config\" name=\"unknown\" type=\"file\"><property key=\"path\" value=\""+tmpDirPath+"otst_unknown.csv\"/></output>";
        sUri = urlBase + "/output/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status when changing the 'unknown' output");
    }

    /**
     * Change the value of the 'C_change_1' constant indicator of the 'pChange' probe
     * - previous value: 1
     * - new value: 100
     * Check the JSR of the 'pChange' probe has this new value
     */
    @Test
    public void changeIndicator() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;

        // Stop the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/stop";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when stoping the probe");
        
        // Change the indicator
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><indicator xmlns=\"org.ow2.jasmine.probe:probe-config\" name=\"C_change_1\" type=\"constant\"><property key=\"value\" value=\"100\"/><property key=\"type\" value=\"int\"/></indicator>";
        sUri = urlBase + "/indicator/C_change_1";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when changing the indicator");
        
        // Restart the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/start";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when starting the probe");

        // Get data from the task 'pChange' and check it
        logger.info("Wait a while to get time to do one collect ...");
        Thread.sleep(9*1000);
        sUri = urlBase + "/data/pChange";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when getting data");
        // Check the results of one collect of the 'pChange' probe
        List<JsrType> jsrs = response.getEntity(new GenericType<List<JsrType>>() {});
        Assert.assertEquals(jsrs.size(), 1, "Incorrect JSR number");
        for (JsrType jsr : jsrs) {
            Assert.assertEquals(jsr.getType(), TypeNameType.INT, "Incorrect JSR type for " + jsr.getName());
            if ("C_change_1".equals(jsr.getName())) {
                Assert.assertEquals(jsr.getValue().getIntvalue().intValue(), 100, "Incorrect JSR value for " + jsr.getName());
            } else {
                Assert.fail("Incorrect JSR name " + jsr.getName());
            }
        }
    }

    /**
     * Change an unknown indicator.
     * We must got an error status.
     */
    @Test
    public void changeUnknownIndicator() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;

        // Change the indicator
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><indicator xmlns=\"org.ow2.jasmine.probe:probe-config\" name=\"unknown\" type=\"constant\"><property key=\"value\" value=\"100\"/><property key=\"type\" value=\"int\"/></indicator>";
        sUri = urlBase + "/indicator/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status when changing the 'unknown' indicator");
        
    }

    /**
     * Change the indicator list of the 'pChange' probe
     * - old indicator: C_change
     * - new indicator: C_data_5
     * Check the JSR of the 'pChange' probe is this new constant indicator with 5 as value
     */
    @Test
    public void changeProbe() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;

        // Stop the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/stop";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when stoping the probe");
        
        // Change the probe
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><probe xmlns=\"org.ow2.jasmine.probe:probe-config\" id=\"pChange\"><indicator>C_data_5</indicator></probe>";
        sUri = urlBase + "/probe/pChange";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when changing the probe");
        
        // Restart the 'pChange' probe
        sUri = urlBase + "/probe/pChange/action/start";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.post(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_ACCEPTED, "Bad HTTP status when starting the probe");

        // Get data from the task 'pChange' and check it
        logger.info("Wait a while to get time to do one collect ...");
        Thread.sleep(9*1000);
        sUri = urlBase + "/data/pChange";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status when getting data");
        // Check the results of one collect of the 'pChange' probe
        List<JsrType> jsrs = response.getEntity(new GenericType<List<JsrType>>() {});
        Assert.assertEquals(jsrs.size(), 1, "Incorrect JSR number");
        for (JsrType jsr : jsrs) {
            Assert.assertEquals(jsr.getType(), TypeNameType.INT, "Incorrect JSR type for " + jsr.getName());
            if ("C_data_5".equals(jsr.getName())) {
                Assert.assertEquals(jsr.getValue().getIntvalue().intValue(), 5, "Incorrect JSR value for " + jsr.getName());
            } else {
                Assert.fail("Incorrect JSR name " + jsr.getName());
            }
        }
    }

    /**
     * Change an unknown probe.
     * We must got an error status.
     */
    @Test
    public void changeUnknownProbe() throws Exception {
        WebResource webResource;
        ClientResponse response;
        String sUri;

        // Change the probe
        String input = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\" ?><probe xmlns=\"org.ow2.jasmine.probe:probe-config\" id=\"unknown\"><indicator>C_data_5</indicator></probe>";
        sUri = urlBase + "/probe/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.type("application/xml").put(ClientResponse.class, input);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status when changing the 'unknown' probe");
    }

}
