/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.correlate.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * @author durieuxp
 */
@Component(name="CorrelateCollectorService")
@Provides
public class CorrelateCollectorService extends JCollectorService {
    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Correlate several source indicator's values.";
    public final static String LONG_DESC =
            "The indicator definition requires the correlate operation and a list of source indicators.";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
        "Source indicators should have their value composed of only one result (current limitation). The result must be a Number." +
        "\nFor some operators (div|sub|percent), only 2 sources are required.";
    public final static int SOURCE_MIN = 2;
    public final static int SOURCE_MAX = 3;

    /**
     * Properties
     */
    public final static String PROP_OP = "op";
    public final static String PROP_OP_DESC =
            "Correlate operation. Must be one of : add|sub|mul|div|percent";

    /**
     * List of supported operations
     */
    public final static int OP_ADD = 1;
    public final static int OP_SUB = 2;
    public final static int OP_MUL = 3;
    public final static int OP_DIV = 4;
    public final static int OP_PERCENT = 5;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="correlate")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(SOURCE_MIN);
        collectorInfo.setSourceMax(SOURCE_MAX);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_OP, PROP_OP_DESC, true));
        collectorInfo.setPropertyInfos(properties);
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe.
     * First parse indicator properties, and get a collector
     * for each source. This allows to 'notify' to sources that an additional
     * 'consumer' collector exists for them. This allows collectors that manage a cache
     * of values to update cache size.
     * @param indicator the indicator for which values are collected by the collector
     * @param probe the probe using the indicator
     * @return the JasmineCollector instance created to collect values for the given indicator
     * within the given probe.
     * @throws JasmineCollectorException The collector could not be created.
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe)
    throws JasmineCollectorException {

        // sanity check, not really needed.
        if (! indicator.getType().equals(indicatorType)) {
            logger.error("Bad type: " + indicator.getType());
            throw new JasmineCollectorException("Bad type: " + indicator.getType());
        }

        Map<String, String> props = indicator.getProperties();

        // Check the operation
        String operation = props.get(PROP_OP);
        if (operation == null) {
            String err = "Operation not defined in " + indicator.getName();
            logger.error(err);
            throw new JasmineCollectorException(err);
        }
        int ope = 0;
        if (operation.equalsIgnoreCase("add")) {
            ope = OP_ADD;
        } else if (operation.equalsIgnoreCase("sub")) {
            ope = OP_SUB;
        } else if (operation.equalsIgnoreCase("mul")) {
            ope = OP_MUL;
        }  else if (operation.equalsIgnoreCase("div")) {
            ope = OP_DIV;
        }  else if (operation.equalsIgnoreCase("percent")) {
            ope = OP_PERCENT;
        } else {
            String err = "Operation not supported in CorrelateCollectorService: " + operation;
            logger.error(err);
            throw new JasmineCollectorException(err);
        }

        // Get the collectors corresponding to the source indicators
        Collection<JasmineCollector> sources = new ArrayList<JasmineCollector>();
        List<String> inames = indicator.getSources();
        if (inames == null) {
            String err = "No sources defined in " + indicator.getName();
            logger.error(err);
            throw new JasmineCollectorException(err);
        }
        for (String iname : inames) {
            JasmineCollector source = null;
            try {
                source = probeManager.getCollector(iname, probe.getId());
            } catch (JasmineProbeException e) {
                logger.error("Cannot get value: " + e);
                throw new JasmineCollectorException(e.getMessage());
            }
            sources.add(source);
        }
        // Create the Collector
        JCollector collector = new CorrelateCollector(probe.getId(), indicator, probe.getPeriod(), ope, sources);
        addCollector(indicator.getName(), collector);
        return collector;
    }

    /**
     * return the List of indicators used by this indicator.
     * @param indic JasmineIndicator
     * @return
     */
    @Override
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        List<String> ret = new ArrayList<String>();
        for (String iname : indic.getSources()) {
            if (!isNumeric(iname)) {
                ret.add(iname);
            }
        }
        return ret;
    }

    // ----------------------------------------------------------------------------------------
    // private methods
    // ----------------------------------------------------------------------------------------

    private boolean isNumeric(String s) {
        return s.matches("((-|\\+)?[0-9]+(\\.[0-9]+)?)+");
    }

}
