/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.management.Attribute;
import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.util.CsvConverter;
import org.ow2.jasmine.probe.util.MetaData;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Base class for regular JmxCollector and JmxCollector for tests.
 * @author danesa
 *
 */
public abstract class JmxCollectorBase extends JCollector {

    protected Log logger = LogFactory.getLog(JmxCollector.class);

    /**
     * Jmx MBean Filter (ObjectName syntax)
     */
    private String pattern;

    public String getPattern() {
        return pattern;
    }

    /**
     * The name of the attributes or attribute fragments to get.
     */
    private List<String> attrlist;

    /**
     * Constructor
     * @param probeId probe Identifier
     * @param indicator JasmineIndicator definition
     * @param period in seconds
     * @param mbean MBean pattern
     * @param csv CSV list of attributes
     */
    public JmxCollectorBase(final String probeId, final JasmineIndicator indicator, final int period, final String mbean, String csv) {
        super(probeId, indicator, period);
        this.pattern = mbean;
        this.attrlist = CsvConverter.csv2list(csv);
    }

    /**
     * Getter for attrlist (list of provided attribute and or fragment names).
     * @return return the provided attribute and or fragment names.
     */
    public List<String> getAttrlist() {
        return attrlist;
    }

    /**
     * @return The ObjectName instance corresponding to the mbean pattern provided in
     * the indicator definition.
     * @throws JasmineCollectorException an ObjectName could not be obtained as the pattern
     * in not well formed given the JMX naming conventions.
     */
    public ObjectName getObjectName() throws JasmineCollectorException {
        return JmxUtil.getObjectName(pattern);
    }


    /**
     * Allows to know if attributes and/or fragments are provided by the indicator definition.
     * @return true if attributes and or fragments are provided by the indicator definition.
     */
    protected boolean hasAttributes() {
        if (attrlist == null || attrlist.isEmpty()) {
            return false;
        } else {
            // if one element equal to "all" in the attrlist
            if (attrlist.size() == 1) {
                for (String elem : attrlist) {
                    if (JmxCollectorService.VALUE_ALL.equals(elem)) {
                        return false;
                    }
                }
            }
            return true;
        }
    }

    /**
     * Create a JIV on the basis of collected metrics.
     * @param name the value name
     * @param probeId the id of the probe
     * @param scale a scale to apply to the value
     * @param metrics the metrics to use to construct the value
     * @return the created JIV
     * @throws JasmineCollectorException
     */
    protected JasmineIndicatorValue getJiv(final String name, final String probeId, final int scale, final List<Metric> metrics) throws JasmineCollectorException {
        logger.debug("Get indicator value for indicator {0} in probe {1}", name, probeId);

        // Create the JasmineIndicatorValue instance to return
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(name);

        // fill in JasmineIndicValue instance with JasmineSingleResults corresponding to metrics
        for (Metric metric : metrics) {
            // Each metric corresponds to a MBean.
            // -----------------------------------

            if (metric.getAttributeList().size() == 0) {
                logger.warn("Result with no value: " + metric.toString());
                continue;
            }

            // JIV's multiValue is set to false only if THE metric's multiValue
            // is false.
            jiv.setMultiValue(metric.isMultiValue());

            Map<String, String> props = metric.getProperties();
            if (!props.isEmpty()) {
                for (Iterator<String> propKeys = props.keySet().iterator(); propKeys.hasNext(); ) {
                    String propKey = (String) propKeys.next();
                    String propVal = props.get(propKey);
                    jiv.addMetadata(propKey, propVal);
                }
            }
            for (Iterator it = metric.getAttributeList().iterator(); it.hasNext(); ) {
                Attribute att = (Attribute) it.next();
                if (att.getValue() == null) {
                    // No value: forget this attribute
                    logger.warn("No value for " + att.getName() + " in indicator: " + getIndicatorName() + " and probe: " + probeId);
                    continue;
                }

                // decompose each attribute into elementary fragments
                List<Attribute> decomposedList;
                try {
                    decomposedList = JmxUtil.getAttributeFragments(att);
                    for (Attribute dAtt : decomposedList) {
                        // TODO add metadata + properties
                        String dAttName = dAtt.getName();
                        Object dAttValue = dAtt.getValue();
                        JasmineSingleResult jsr = getJsr(dAttName, dAttValue, metric.getTimestamp(), indicator.getScale());
                        if (jsr != null) {
                            // Add MBean name as "mbean" property
                            jsr.addProperty(MetaData.MBEAN, metric.getMBean().toString());
                            jiv.addValue(jsr);
                        }
                    }
                } catch (ComplexValuesException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }

            }
        }

        /**
         * All the Metrics have the same values for the following metadata :
         * target
         * META_SERVER - already set in Metric.properties
         * META_DOMAIN - already set in Metric.properties
         * META_URL    - already set in Metric.properties
         * Use the first Metric instance from the metrics to set metadata into the jiv.
         */
        Metric m = metrics.get(0);
        jiv.setTarget(m.getTarget());
        String key = MetaData.SERVER;
        jiv.addMetadata(key, m.getProperties().get(key));
        key = MetaData.DOMAIN;
        jiv.addMetadata(key, m.getProperties().get(key));
        key = MetaData.URL;
        jiv.addMetadata(key, m.getProperties().get(key));
        return jiv;
    }
}
