/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.linux.internal;

import java.util.ArrayList;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * This CollectorService implements the "linux" collector.
 * @author danesa
 */
@Component(name="LinuxCollectorService")
@Provides
public class LinuxCollectorService extends JCollectorService {
    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Get Linux system resource parameters.";
    public final static String LONG_DESC =
            "Resources can be: kernel, .... \n" +
            "Parameters depend on the resource.";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "No source indicator used.";
    public final static int SOURCE_MIN = 0;
    public final static int SOURCE_MAX = 0;

    /**
     * Properties
     */
    public final static String PROP_RESOURCE = "resource";
    public final static String PROP_RESOURCE_DESC =
            "Resource type. One of disk,memory,network,kernel,cpu (cpu for aggregated values, or cpui (cpu0 etc.) for one of the possible cpus";
    public final static String PROP_PARAMS = "params";
    public final static String PROP_PARAMS_DESC =
            "Comma separated list of parameters to poll.\n" +
            "For kernel resource, available parameters are 'intr' (interrupts count), " +
            "'ctxt' (number of context switches), 'processes' (number of created processes and threads)\n" +
            "For cpu resource, available parameters are 'user', 'nice', 'kernel' and 'idle'\n ";

    /**
     * Supported Linux versions
     */
    protected int osType;
    public final static int LINUX_2_4 = 10;
    public final static int LINUX_2_6 = 11;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="linux")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(0);
        collectorInfo.setSourceMax(0);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_RESOURCE, PROP_RESOURCE_DESC, true));
        properties.add(new JasminePropertyInfo(PROP_PARAMS, PROP_PARAMS_DESC, false));
        collectorInfo.setPropertyInfos(properties);

        // Check System version
        if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
            if (System.getProperty("os.version").compareTo("2.6") >= 0) {
                // Assumes that versions newer versions will be as 2.6
                osType = LINUX_2_6;
            } else {
                // Assumes that other versions will be as 2.4
                osType = LINUX_2_4;
            }
        }
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------
    /**
     * Create a Collector for this JasmineIndicator
     * @param indicator
     * @param probe
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    @Override
    public JasmineCollector getCollector(JasmineIndicator indicator,
            JasmineProbe probe) throws JasmineCollectorException {
        JCollector collector = new LinuxCollector(probe.getId(), indicator, probe.getPeriod());
        collector.setProbeManager(probeManager);
        addCollector(indicator.getName(), collector);
        return collector;
    }

}
