/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.df.internal;

import java.util.ArrayList;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * This CollectorService implements the "df" unix command.
 * Works only on linux like system, since the df command must exist.
 * @author durieuxp
 */
@Component(name="DfCollectorService")
@Provides
public class DfCollectorService extends JCollectorService {
    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Make a df command (linux only)";
    public final static String LONG_DESC =
            "Get the used space on a disk, in percent. If no disk is specified, all disks are checked.";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "No source indicator used.";
    public final static int SOURCE_MIN = 0;
    public final static int SOURCE_MAX = 0;

    /**
     * Properties
     */
    public final static String PROP_DISK = "disk";
    public final static String PROP_DISK_DESC =
            "device to check (pathname relative to /dev). Example: sda1. If not specified: all the devices are checked.";

    /**
     * The name of the property allowing to get disk name.
     */

    /**
     * DiskFree thread collector
     * In this implementation, we have only one thread.
     */
    protected DiskFree df = null;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="df")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(SOURCE_MIN);
        collectorInfo.setSourceMax(SOURCE_MAX);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_DISK, PROP_DISK_DESC, false));
        collectorInfo.setPropertyInfos(properties);

        // Create the DiskFree ThreadCollector
        df = new DiskFree();
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
        // Stop DiskFree
        df.remove();
        df = null;
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for this JasmineIndicator.
     * Called by the JasmineProbeManager when first starting a probe
     * (at the probe creation).
     * @param indicator contains the indicator's parameters.
     * @param probe the probe that gets the collector
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe) throws JasmineCollectorException {
        JCollector collector = new DfCollector(probe.getId(), indicator, probe.getPeriod());
        collector.setProbeManager(probeManager);
        addCollector(indicator.getName(), collector);
        df.addCollector((DfCollector) collector);
        return collector;
    }

}
