/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.linux.internal;

import java.util.Map;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.linux.LinuxCollectingException;
import org.ow2.jasmine.probe.collectors.linux.LinuxResourceCollector;
import org.ow2.jasmine.probe.collectors.linux.NoProcResourceException;
import org.ow2.jasmine.probe.collectors.linux.NoSuchLinuxResourceException;
import org.ow2.jasmine.probe.util.CsvConverter;
import org.ow2.jasmine.probe.util.MetaData;

/**
 * Collector allowing to get measurements about a Linux OS:
 * kernel activity, system memory, devices mounted etc.
 * @author danesa
 *
 */
public class LinuxCollector extends JCollector {

    /**
     * System resource types : kernel, network, disk, memory, cpu
     */
    protected final String KERNEL =  "kernel";
    protected final String DISK =    "disk";
    protected final String MEMORY =  "memory";
    protected final String CPU =     "cpu";
    protected final String NETWORK = "network";

    protected final String PROC_STAT_FILE = "/proc/stat";

    /**
     * The resource type of this collector.
     */
    protected String resourceType;

    /**
     * String Array of resource parameters to be polled
     */
    protected String[] resourceParams = null;

    /**
     * Ids corresponding to the resources to be polled
     */
    protected int[] rids = null;

    /**
     * Specific collector for the given resource type
     */
    private LinuxResourceCollector linuxCollector = null;

    public LinuxCollector(String probeId, JasmineIndicator def, int period) throws JasmineCollectorException {
        super(probeId, def, period);
        // specific part of JasmineIndicator (property list)
        Map<String, String> props = indicator.getProperties();
        this.resourceType = props.get(LinuxCollectorService.PROP_RESOURCE);
        if (resourceType == null) {
            throw new JasmineCollectorException("No resource defined for linux indicator " + def.getName() + ". Use "
                    + LinuxCollectorService.PROP_RESOURCE + " property");
        }

        this.resourceParams = CsvConverter.csv2array(props.get(LinuxCollectorService.PROP_PARAMS));
        if (resourceParams == null) {
            throw new JasmineCollectorException("No parameters defined for linux indicator " + def.getName() + ". Use "
                    + LinuxCollectorService.PROP_PARAMS + " property");
        }
        int nbParams = resourceParams.length;

        if (KERNEL.equals(resourceType)) {
            try {
                linuxCollector = new KernelCollector(KERNEL, PROC_STAT_FILE);
            } catch (NoProcResourceException e1) {
                throw new JasmineCollectorException(e1.toString());
            }
        } else if (resourceType.startsWith(CPU)) {
            try {
                linuxCollector = new CpuCollector(resourceType, PROC_STAT_FILE);
            } catch (NoProcResourceException e) {
                throw new JasmineCollectorException(e.toString());
            }
        } else {
            throw new JasmineCollectorException(resourceType + " resource not supported by linux indicators");
        }

        this.rids = new int[nbParams];
        for (int i = 0; i < nbParams; i++) {
            try {
                rids[i] = linuxCollector.getParamId(resourceParams[i]);
            } catch (NoSuchLinuxResourceException e) {
                throw new JasmineCollectorException(e.toString());
            }
        }
    }

    @Override
    public JasmineIndicatorValue getLastResult()
            throws JasmineCollectorException {
        // Get values from the collector
        try {
            // keep same timestamp for all parameters polling
            long time = System.currentTimeMillis();
            long[] values = linuxCollector.getValue(rids);

            JasmineIndicatorValue jiv = new JasmineIndicatorValue();
            jiv.setName(indicator.getName());

            // Set domain and server names in metadata
            jiv.addMetadata(MetaData.SERVER, getServerName());
            jiv.addMetadata(MetaData.DOMAIN, getDomainName());
            // Set target in JIV
            jiv.setTarget(getDefaultTarget());
            //
            jiv.addMetadata(LinuxCollectorService.PROP_RESOURCE, this.resourceType);
            for (int i = 0; i < values.length; i++) {
                long value = values[i];
                if (indicator.getScale() != 1) {
                    value = value / indicator.getScale();
                }
                String name = linuxCollector.getParamName(rids[i]);
                // Build a result
                JasmineSingleResult jsr = new JasmineSingleNumberResult();
                jsr.setName(name);
                jsr.setValue(value);
                jsr.setTimestamp(time);
                jsr.addProperty(LinuxCollectorService.PROP_PARAMS,  name);
                // Add this result to the JasmineIndicatorValue
                jiv.addValue(jsr);
            }
            if (resourceParams.length == 1) {
                jiv.setMultiValue(false);
            }
            return jiv;
        } catch (LinuxCollectingException e) {
           throw new JasmineCollectorException(e.toString());
        } catch (NoSuchLinuxResourceException e) {
            throw new JasmineCollectorException(e.toString());
        }
    }

    @Override
    public void startPolling() {
        // nothing to do
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
    }

    @Override
    public void stopPolling() {
        // nothing to do
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
    }

}
