/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.api.generated.ErrorType;
import org.ow2.jasmine.probe.api.generated.Link;
import org.ow2.jasmine.probe.api.generated.ProbeConfType;
import org.ow2.jasmine.probe.api.generated.ProbeResource;
import org.ow2.jasmine.probe.api.generated.ProbeType;
import org.ow2.jasmine.probe.api.generated.Status;
import org.ow2.jasmine.probe.api.generated.StatusType;
import org.ow2.jasmine.probe.api.generated.TaskType;
import org.ow2.jasmine.probe.rest.DataTasks;
import org.ow2.jasmine.probe.rest.Indicators;
import org.ow2.jasmine.probe.rest.Outputs;
import org.ow2.jasmine.probe.rest.Probes;
import org.ow2.jasmine.probe.rest.Targets;
import org.ow2.jasmine.probe.rest.TasksManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implements REST API for probes management : org.ow2.jasmine.probe.rest.Probes
 * @author danesa
 *
 */
public class JProbes implements Probes {

    private Log logger = LogFactory.getLog(this.getClass());

    private JasmineProbeManager jpManager = null;

    private TasksManager taskManager = null;

    private static final String REST_OUTPUT_TYPE = "rest";
    private static final String REST_OUTPUT_PREFIX = "rest_out_";
    private static final String REST_OUTPUT_PROP = "taskId";

    public JProbes(JasmineProbeManager jpManager, TasksManager taskManager) {
        this.jpManager = jpManager;
        this.taskManager = taskManager;
    }

    @Override
    /**
     * GET http://host:port/jprobe/probe
     * Get the list of all probes
     */
    public Response	getProbes(UriInfo ui) {

        List<JAXBElement<ProbeResource>> elementsList = new ArrayList<JAXBElement<ProbeResource>>();

        List<JasmineProbe> probesList = jpManager.getProbes();
        for (JasmineProbe probe : probesList) {
            ProbeResource probeElement = createSimpleProbeResource(probe, ui);
            QName name  = new QName("org.ow2.jasmine.probe:probe-config", "probe");
            JAXBElement<ProbeResource> xmlElement =  new JAXBElement<ProbeResource>(name, ProbeResource.class, probeElement);
            elementsList.add(xmlElement);
        }

        return Response
                        .status(Response.Status.OK)
                        .entity(new GenericEntity<List<JAXBElement<ProbeResource>>>(elementsList) {})
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    /**
     * GET http://host:port/jprobe/probe/{probeId}
     * Get a probe's definition and status
     */
    public Response getProbe(String probeId, UriInfo ui) {
        JasmineProbe jprobe;
        try {
            jprobe = jpManager.getProbe(probeId);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        ProbeResource probe = createProbeResource(jprobe, ui);

        QName name  = new QName("org.ow2.jasmine.probe:probe-config", "probe");
        JAXBElement<ProbeResource> probeResource =  new JAXBElement<ProbeResource>(name, ProbeResource.class, probe);

        return Response
                        .status(Response.Status.OK)
                        .entity(probeResource)
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    /**
     * POST http://host:port/jprobe/probe
     * Create a new probe - corresponds to probe-create command.
     * The probeDef may contain the new probe's id.
     * Otherwise, the id is generated by JasmineProbe.
     * @param probeDef new probe's definition
     * @return status may be
     * - CRETED (201) if creation successful (in this case the response contains the new probe's URI)
     * - BAD_REQUEST (400) if failure (for example trying to create a probe while the provided id exists)
     */
    @Override
    public Response createProbe(ProbeType probeDef) {
        JasmineProbe probe = new JasmineProbe();
        probe.setId(probeDef.getId());
        probe.setPeriod(probeDef.getPeriod());
        probe.setIndicatorList(probeDef.getIndicator());
        probe.setOutputList(probeDef.getOutput());
        probe.setTargetList(probeDef.getTarget());

        boolean toStart = false;
        if (probeDef.getStatus() != null) {
            if (probeDef.getStatus().equals(StatusType.STARTED)) {
                probe.setStatus(JasmineProbe.PROBE_TOSTART);
                toStart = true;
            }
        }

        String probeId = null;
        try {
            probeId = jpManager.createProbe(probe);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        // the location of the created probe resource
        URI location = UriBuilder.fromPath(probeId).build();

        if (toStart) {
            try {
                jpManager.startProbe(probeId);
                return Response.status(Response.Status.ACCEPTED).header("Location", location).build();
            } catch (JasmineProbeException e) {
                return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
            }
        }

        return Response.created(location).build();
    }

    /**
     * PUT http://host:port/jprobe/probe/{probeId}
     * Change a probe
     * @param probeId the id of the probe to update
     * @param probeDef contains the updated configuration
     * @return status may be
     * - OK (200) if change successful
     * - NOT_FOUND if no probe with probeId exists
     */
    @Override
    public Response changeProbe(ProbeConfType probeDef, String probeId) {

        try {
            jpManager.getProbe(probeId);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        JasmineProbe probe = new JasmineProbe();
        probe.setId(probeId);
        Integer intPeriod = probeDef.getPeriod();
        if (intPeriod == null) {
            // don't change period
            probe.setPeriod(0);
        } else {
            probe.setPeriod(intPeriod.intValue());
        }
        probe.setIndicatorList(probeDef.getIndicator());
        probe.setOutputList(probeDef.getOutput());
        probe.setTargetList(probeDef.getTarget());

        try {
            jpManager.changeProbe(probe);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response changeProbePeriod(String probeId, String newPeriod) {
        int p;
        try {
            p = Integer.parseInt(newPeriod);
        } catch (NumberFormatException e) {
            ErrorType err = new ErrorType();
            err.setMessage(e.toString());
            QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "error");
            JAXBElement<ErrorType> errElem = new JAXBElement<ErrorType>(qname, ErrorType.class, err);
            return Response
            .status(Response.Status.BAD_REQUEST)
            .entity(errElem)
            .type(MediaType.APPLICATION_XML)
            .build();
        }

        try {
            jpManager.changeProbePeriod(probeId, p);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response deleteProbe(String probeId) {
        try {
            jpManager.removeProbe(probeId);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }
        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response setState(String probeId, String actionName) {
        try {
            if (START_ACTION.equals(actionName)) {
                jpManager.startProbe(probeId);
            }
            if (STOP_ACTION.equals(actionName)) {
                jpManager.stopProbe(probeId);
            }
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.INTERNAL_SERVER_ERROR);
        }
        return Response.status(Response.Status.ACCEPTED).build();
    }


    @Override
    public Response getDataTask(String probeId, UriInfo ui) {
        JasmineProbe probe = null;
        try {
            probe = jpManager.getProbe(probeId);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        // Create task for probe probeId, or return task id if already created
        String taskId = taskManager.createTask(probeId);

        JasmineOutput jor = getRestOutput(taskId);
        if (jor == null) {
            return RestUtil.errorResponse(
                    new JasmineProbeException("Cannot create task for probe " + probeId + " because cannot create REST output " + REST_OUTPUT_PREFIX + taskId),
                    Response.Status.INTERNAL_SERVER_ERROR);
        }

        try {
            jpManager.createOutput(jor);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.INTERNAL_SERVER_ERROR);
        }

        try {
            probe = jpManager.getProbe(probeId);
            List<String> outputList = probe.getOutputList();
            if (!outputList.contains(jor.getName())) {
                outputList.add(jor.getName());
                probe.setOutputList(outputList);
                jpManager.changeProbe(probe);
            }
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.INTERNAL_SERVER_ERROR);
        }

        URI taskUri = ui.getBaseUriBuilder().path(DataTasks.RESOURCE_NAME + "/" + taskId).build();

        TaskType taskElement = new TaskType();
        taskElement.setHref(taskUri.toString());
        taskElement.setId(taskId);
        QName name  = new QName("org.ow2.jasmine.probe:probe-config", "data");
        JAXBElement<TaskType> xmlElement =  new JAXBElement<TaskType>(name, TaskType.class, taskElement);

        return Response
                 .status(Response.Status.ACCEPTED)
                 .header("Location", taskUri)
                 .entity(xmlElement)
                 .type(MediaType.APPLICATION_XML_TYPE)
                 .build();
    }

    @Override
    public Response deleteDataTask(String probeId) {
        JasmineProbe probe = null;
        try {
            probe = jpManager.getProbe(probeId);
            List<String> outList = probe.getOutputList();
            JasmineOutput out = null;
            String outId = null;
            for (String outputId : outList) {
                out = jpManager.getOutput(outputId);
                if (REST_OUTPUT_TYPE.equals(out.getType())) {
                    // get taskId from "taskId" property
                    String taskId = out.getProperties().get(REST_OUTPUT_PROP);
                    if (taskId != null) {
                        // found the corresponding REST output
                        outId = outputId;
                        // remove task with taskId
                        taskManager.removeTask(taskId);
                        break;
                    }
                }
            }
            if (outId != null) {
                // remove the output from the outputList (change probe)
                // remove the output
                List<String> updatedList = new ArrayList<String>(outList);
                updatedList.remove(outId);
                probe.setOutputList(updatedList);
                jpManager.removeOutput(outId);
            }
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        return Response.status(Response.Status.OK).build();
    }


    // =========== private methods =============
    /**
     * Create a simple ProbeResource (only Id and URI set) for a probe.
     * @param probe the JasmineProbe object
     * @param ui UriInfo from context
     */
    private ProbeResource createSimpleProbeResource(final JasmineProbe probe, final UriInfo ui) {
        ProbeResource probeResource = new ProbeResource();
        String probeId = probe.getId();
        URI probeUri = ui.getAbsolutePathBuilder().path(probeId).build();
        probeResource.setId(probe.getId());
        probeResource.setHref(probeUri.toString());
        return probeResource;
    }

    /**
     * Create a ProbeResource for a probe.
     * @param probe the JasmineProbe object
     * @param ui UriInfo from context
     */
    private ProbeResource createProbeResource(final JasmineProbe probe, final UriInfo ui) {
        ProbeResource probeResource = createSimpleProbeResource(probe, ui);
        // Get base URI from current probe's URI
        UriBuilder ub = ui.getAbsolutePathBuilder();
        String probePath = ui.getPath();
        String probeUri = ub.build("").toString();
        String probeBaseUri = probeUri.substring(0, probeUri.lastIndexOf(probePath));
        String indicatorsUri = probeBaseUri + Indicators.RESOURCE_NAME;
        String outputsUri = probeBaseUri + Outputs.RESOURCE_NAME;
        String targetsUri = probeBaseUri + Targets.RESOURCE_NAME;
        // use current probeUri
        probeResource.setHref(probeUri);
        // set period
        Integer value = Integer.valueOf(probe.getPeriod());
        probeResource.setPeriod(value);
        // set indicator names and links
        List<Link> links = new ArrayList<Link>();
        List<String> indicators = new ArrayList<String>();
        List<String> indicNames = probe.getIndicatorList();
        updateListAndLinks(indicNames, indicators, links, indicatorsUri);
        // set output names and links
        List<String> outputs = new ArrayList<String>();
        List<String> outNames = probe.getOutputList();
        updateListAndLinks(outNames, outputs, links, outputsUri);
        // set target names and links
        List<String> targets = new ArrayList<String>();
        List<String> targetNames = probe.getTargetList();
        updateListAndLinks(targetNames, targets, links, targetsUri);
        probeResource.setLink(links);
        probeResource.setIndicator(indicators);
        probeResource.setOutput(outputs);
        probeResource.setTarget(targets);
        // set status
        Status status = null;
        int statusCode = probe.getStatus();
        switch (statusCode) {
        case JasmineProbe.PROBE_RUNNING:
            status = Status.RUNNING;
            break;
        case JasmineProbe.PROBE_STARTED:
        case JasmineProbe.PROBE_TOSTART:
            status = Status.STARTED;
            break;
        case JasmineProbe.PROBE_FAILED:
            status = Status.FAILED;
            break;
        case JasmineProbe.PROBE_STOPPED:
            status = Status.STOPPED;
            break;
        default:
            status = Status.INVALID;
            break;
        }
        probeResource.setStatus(status);
        return probeResource;
    }

    /**
     * Update links and name lists for a probe representation
     * @param nameList the list of element names (indicator, output, target names)
     * @param updateList the list to update
     * @param links the links to update
     * @param path the path corresponding to the element type
     */
    private void updateListAndLinks(final List<String> nameList, List<String> updateList, List<Link> links, final String path) {
        for (String name : nameList) {
            Link link = new Link();
            link.setRel("down");
            UriBuilder builder = UriBuilder.fromPath(path);
            URI uri = builder.path(name).build();
            link.setHref(uri.toString());
            links.add(link);
            updateList.add(name);
        }
    }

    /**
     * Return the REST output corresponding to the default name if it exists.
     * Otherwise try to create a new JasmineOutput with the default name.
     * @param taskId Used to construct the default name for the rest output
     * @return a JasmineOutput associated with the task and with the a probe.
     */
    private JasmineOutput getRestOutput(final String taskId) {
        String defaultId = REST_OUTPUT_PREFIX + taskId;
        try {
            JasmineOutput output = jpManager.getOutput(defaultId);
            // check rest output
            if (REST_OUTPUT_TYPE.equals(output.getType()) &&
                    output.getProperties().containsKey(REST_OUTPUT_PROP) &&
                    taskId.equals(output.getProperties().get(REST_OUTPUT_PROP))) {
                return output;
            } else {
                return null;
            }
        } catch (JasmineProbeException e) {
            // there is no output with defaultId
            return createRestOutput(defaultId, taskId);
        }
    }

    private JasmineOutput createRestOutput(final String outputId, final String taskId) {
        JasmineOutput creaOutput = new JasmineOutput();
        creaOutput.setName(outputId);
        creaOutput.setType(REST_OUTPUT_TYPE);
        Map<String, String> creaOutProps = new HashMap<String, String>();
        creaOutProps.put(REST_OUTPUT_PROP, taskId);
        creaOutput.setProperties(creaOutProps);
        return creaOutput;
    }

}
