/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.api.generated.ErrorType;
import org.ow2.jasmine.probe.api.generated.PropType;
import org.ow2.jasmine.probe.api.generated.TargetBaseType;
import org.ow2.jasmine.probe.api.generated.TargetResource;
import org.ow2.jasmine.probe.rest.Targets;

public class JTargets implements Targets {

    private JasmineProbeManager jpManager = null;

    public JTargets(JasmineProbeManager jpManager) {
        this.jpManager = jpManager;
    }

    @Override
    public Response getTarget(String name, UriInfo ui) {
        UriBuilder ub = ui.getAbsolutePathBuilder();
        String targetUri = ub.build("").toString();
        String baseUri = targetUri.substring(0, targetUri.lastIndexOf(name));

        JasmineTarget jtarget;
        try {
            jtarget = jpManager.getTarget(name);
        } catch (JasmineProbeException e) {
            ErrorType err = new ErrorType();
            err.setMessage(e.getMessage());
            QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "error");
            JAXBElement<ErrorType> errElem = new JAXBElement<ErrorType>(qname, ErrorType.class, err);
            return Response
            .status(Response.Status.NOT_FOUND)
            .entity(errElem)
            .type(MediaType.APPLICATION_XML)
            .build();
        }
        if (jtarget == null) {
            ErrorType err = new ErrorType();
            err.setMessage("Cannot find target with name: " + name);
            QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "error");
            JAXBElement<ErrorType> errElem = new JAXBElement<ErrorType>(qname, ErrorType.class, err);
            return Response
            .status(Response.Status.NOT_FOUND)
            .entity(errElem)
            .type(MediaType.APPLICATION_XML)
            .build();
        }

        TargetResource target = new TargetResource();
        target.setName(name);
        target.setHref(targetUri);

        List<PropType> property = new ArrayList<PropType>();
        for (String propKey : jtarget.getProperties().keySet()) {
            String propValue = jtarget.getProperties().get(propKey);
            PropType prop = new PropType();
            prop.setValue(propValue);
            prop.setKey(propKey);
            property.add(prop);
        }
        target.setProperty(property);

        QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "target");
        JAXBElement<TargetResource> targetResource = new JAXBElement<TargetResource>(qname, TargetResource.class, target);

        return Response
                        .status(Response.Status.OK)
                        .entity(targetResource)
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    public Response getTargets(boolean verbose, UriInfo ui) {
        List<JAXBElement<TargetResource>> elementsList = new ArrayList<JAXBElement<TargetResource>>();

        List<JasmineTarget> targetList = jpManager.getTargets();
        for (JasmineTarget target : targetList) {
                String targetName = target.getName();
                TargetResource targetElement = new TargetResource();
                targetElement.setName(target.getName());

                URI targetUri = ui.getAbsolutePathBuilder().path(targetName).build();
                targetElement.setHref(targetUri.toString());

                if (verbose) {
                    // TODO
                }

                QName name  = new QName("org.ow2.jasmine.probe:probe-config", "target");
                JAXBElement xmlElement =  new JAXBElement<TargetResource>(name, TargetResource.class, targetElement);
                elementsList.add(xmlElement);
        }

        return Response
                        .status(Response.Status.OK)
                        .entity(new GenericEntity<List<JAXBElement<TargetResource>>>(elementsList) {})
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    public Response deleteTarget(String name) {
        try {
            jpManager.removeTarget(name);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }
        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response createTarget(TargetBaseType targetDef) {

        JasmineTarget target = new JasmineTarget();
        target.setName(targetDef.getName());
        target.setType(targetDef.getType());

        Map<String, String> properties = new HashMap<String, String>();

        for (PropType prop : targetDef.getProperty()) {
            properties.put(prop.getKey(), prop.getValue());
        }
        target.setProperties(properties);

        try {
            jpManager.createTarget(target);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();
    }

}
