/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.util.List;

import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;

/**
 * JmxCollector for tests. Doesn't use JmxWorker to collect data from MBeans.
 * It uses a private JmxProxy object for jmx interaction, but this object is not threaded.
 * @author danesa
 *
 */
public class JmxCollectorSimple extends JmxCollectorBase {

    /**
     * ObjectName corresponding to the pattern within indicator's definition.
     */
    private ObjectName objname = null;

    /**
     * Metrics obtained for an indicator.
     */
    private List<Metric> metrics = null;

    /**
     * Number of collectors for a given indicator (the current one) and a given probe.
     * By default, one collector for an indicator and a probe.
     * If the same indicator is used several times in that probe, nb will be incremented to the number of usages
     */
    protected int nb = 1;

    /**
     * Allows to return once the response (the JIV) calculated from metrics, even when nb > 1.
     */
    private int nbGetLastResult;

    /**
     * One proxy by target.
     * Currently only one target is supported.
     */
    private JmxProxy jmxProxy = null;

    /**
     * Constructor
     * @param probeId probe Identifier
     * @param indicator JasmineIndicator definition
     * @param period in seconds
     * @param mbean MBean pattern
     * @param csv CSV list of attributes
     * @throws JasmineCollectorException
     */
    public JmxCollectorSimple(String probeId, JasmineIndicator indicator, int period, String mbean, String csv) throws JasmineCollectorException {
        super(probeId, indicator, period, mbean, csv);
        objname = getObjectName();
        jmxProxy = new JmxProxy();
    }

    @Override
    /**
     * Construct the indicator value using collected metrics.
     */
    public JasmineIndicatorValue getLastResult()
            throws JasmineCollectorException {
        JasmineIndicatorValue resultValue = null;

        if (metrics == null) {
            // first call
            try {
                metrics = jmxProxy.collectMetrics(objname, getIndicatorName(), getAttrlist());
            } catch (JmxConnectionException e) {
                // could not get metrics this time
                // (empty metrics)
            }
            nbGetLastResult = nb;
        }

        if (nbGetLastResult > 0) {
            if (metrics == null) {
                logger.warn("Cannot return value for indicator {0} (no available metrics)", getIndicatorName());
                return null;
            }
            if (metrics.isEmpty()) {
                logger.warn("No metrics for indicator {0} in probe {1}", getIndicatorName(), getProbeId());
                return null;
            }
            resultValue = getJiv(getIndicatorName(), getProbeId(), getIndicator().getScale(), metrics);
            logger.debug("Return result for indicator {0} in probe {1} with timestamp {2}", getIndicatorName(), getProbeId(), resultValue.getTimestamp());
            nbGetLastResult--;
            if (nbGetLastResult == 0) {
                metrics = null;
            }
        }
        return resultValue;
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        // nothing to do
        logger.debug("Start " + getIndicatorName() + " in probe " + getProbeId());
    }

    /**
     * Stop polling
     */
    @Override
    public void stopPolling() {
        // nothing to do
        logger.debug("Stop " + getIndicatorName() + " in probe " + getProbeId());
    }

    /**
     * Mark collector as removed.
     * Called when a JasmineCollectorService stops.
     */
    @Override
    public void remove() {
        super.remove();
        jmxProxy.resetMBeanServerConnection();
        jmxProxy = null;
    }


    protected void setJmxConnectionService(JmxConnectionService jmxConnectionService) {
        jmxProxy.setJmxConnectionService(jmxConnectionService);
    }

    protected void setTarget(JasmineTarget target) {
        jmxProxy.setTarget(target);
    }
}
