/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.linux.internal;

import org.ow2.jasmine.probe.collectors.linux.LinuxCollectingException;
import org.ow2.jasmine.probe.collectors.linux.NoProcResourceException;

/**
 * Collector allowing to get measurements about a linux kernel activity.
 * Based on Lewys projetc's KernelProbe.
 * Uses data from /proc/stat file
 *
 * @author danesa
 *
 */
public class KernelCollector extends ResourceCollector {

    // Kernel activity parameters that are supported by the current implementation
    // ---------------------------------------------------------------------------
    // == counts of interrupts serviced since boot time
    private static final String INTERRUPTS = "intr";
    // == total number of context switches across all CPUs
    private static final String CTXT = "ctxt";
    // == the number of processes and threads created
    private static final String PROCESSES = "processes";

    // nb of supported parameters
    private static final int NB_OF_PARAMS  = 3;


    public KernelCollector(final String resourceName, final String fileName) throws NoProcResourceException {
        super(resourceName, fileName, NB_OF_PARAMS);
        params[0] = INTERRUPTS;
        params[1] = CTXT;
        params[2] = PROCESSES;
        checkFormat(buffer);
    }

    /**
     * Verify the expected format:
     * ...........
     * intr value ...
     * ctx value
     * .. (btime line) <-- skipped
     * process value ..
     * ...........
     * @param buffer
     * @throws NoProcResourceException
     */
    private void checkFormat (final byte[] buffer) throws NoProcResourceException {
        int i = 0;
        // Look for INTERRUPTS
        // Skip all content until we have a line starting with 'i' for 'intr'
        do {
            while (buffer[i] != '\n')
                i++;
        } while (buffer[++i] != 'i');
        // Look for 'intr'
        i++;
        if (buffer[i++] != 'n')
            throw invalidFileFormat;
        if (buffer[i++] != 't')
            throw invalidFileFormat;
        if (buffer[i++] != 'r')
            throw invalidFileFormat;
        // Skip spaces
        while (buffer[i] == ' ')
            i++;
        // Skip values
        while ((buffer[i] >= '0') && (buffer[i] <= '9'))
            i++;
        if ((buffer[i] != ' ') && (buffer[i] != '\n') && (buffer[i] != '\r'))
            throw invalidFileFormat;
        // Skip to the next line
        while (buffer[i] != '\n')
            i++;
        i++; // Consume '\n'
        // Look for CTX
        if (buffer[i] != 'c') {
           // Skip disk_io for 2.4 kernel
            while (buffer[i] != '\n')
                i++;
            i++; // Consume '\n'
        }
        if (buffer[i++] != 'c')
            throw invalidFileFormat;
        if (buffer[i++] != 't')
            throw invalidFileFormat;
        if (buffer[i++] != 'x')
            throw invalidFileFormat;
        if (buffer[i++] != 't')
            throw invalidFileFormat;
        // Skip spaces
        while (buffer[i] == ' ')
            i++;
        // Skip value
        while ((buffer[i] >= '0') && (buffer[i] <= '9'))
            i++;
        if ((buffer[i] != ' ') && (buffer[i] != '\n') && (buffer[i] != '\r'))
            throw invalidFileFormat;
        // Skip to the next line
        while (buffer[i] != '\n')
            i++;
        i++; // Consume '\n'
        // Skip btime line
        while (buffer[i] != '\n')
            i++;
        i++; // Consume '\n'
        // Look for PROCESS
        if (buffer[i++] != 'p')
            throw invalidFileFormat;
        if (buffer[i++] != 'r')
            throw invalidFileFormat;
        if (buffer[i++] != 'o')
            throw invalidFileFormat;
        if (buffer[i++] != 'c')
            throw invalidFileFormat;
        if (buffer[i++] != 'e')
            throw invalidFileFormat;
        if (buffer[i++] != 's')
            throw invalidFileFormat;
        if (buffer[i++] != 's')
            throw invalidFileFormat;
        if (buffer[i++] != 'e')
            throw invalidFileFormat;
        if (buffer[i++] != 's')
            throw invalidFileFormat;
        // Skip spaces
        while (buffer[i] == ' ')
            i++;
        while ((buffer[i] >= '0') && (buffer[i] <= '9'))
            i++;
        if ((buffer[i] != ' ') && (buffer[i] != '\n') && (buffer[i] != '\r'))
            throw invalidFileFormat;

    }

    /**
     * Read parameters' values from the file/buffer and set values[].
     * @throws LinuxCollectingException
     */
    protected void getUsage() throws LinuxCollectingException {
        try {
            procFile.seek(0);
            procFile.read(buffer);
            int i = 0;
            long foo;
            // Skip all content until we have a line starting with 'i' for 'intr'
            do {
                while (buffer[i] != '\n')
                    i++;
            } while (buffer[++i] != 'i');
            // Skip 'intr'
            i += INTERRUPTS.length();
            // Get intr value
            while (buffer[i] == ' ')
                i++;
            foo = 0;
            while (buffer[i] >= '0' && buffer[i] <= '9')
            {
                foo = foo * 10 + buffer[i] - '0';
                i++;
            }
            values[0] = foo;
            // Skip end of line, and next line which is disk_io
            while (buffer[i] != '\n')
                i++;
            if (buffer[++i] != 'c') { // Skip disk_io line
                while (buffer[i] != '\n')
                    i++;
                i++;
            }
            // Skip 'ctxt'
            i += CTXT.length();
            // Get ctxt value
            while (buffer[i] == ' ')
                i++;
            foo = 0;
            while (buffer[i] >= '0' && buffer[i] <= '9')
            {
                foo = foo * 10 + buffer[i] - '0';
                i++;
            }
            values[1] = foo;
            // Skip end of line
            while (buffer[i] != '\n')
                i++;
            i++;
            // Skip the next line which is 'btime'
            while (buffer[i] != '\n')
                i++;
            i++;
            // Skip 'processes'
            i += PROCESSES.length();
            // Get processes
            while (buffer[i] == ' ')
                i++;
            foo = 0;
            while (buffer[i] >= '0' && buffer[i] <= '9')
            {
                foo = foo * 10 + buffer[i] - '0';
                i++;
            }
            values[2] = foo;
        } catch (Exception e) {
            throw new LinuxCollectingException(e);
        }
    }
}
