/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.linux.internal;

import java.io.IOException;
import java.io.RandomAccessFile;

import org.ow2.jasmine.probe.collectors.linux.LinuxCollectingException;
import org.ow2.jasmine.probe.collectors.linux.LinuxResourceCollector;
import org.ow2.jasmine.probe.collectors.linux.NoProcResourceException;
import org.ow2.jasmine.probe.collectors.linux.NoSuchLinuxResourceException;
/**
 * Common code for accessing data in /proc/ files.
 *
 * Some notes for using /proc as suggested in Lewys:
 * <p>
 * 1. open the file only once
 * <p>
 * 2. read the whole file at once since every read call on /proc makes the
 * kernel generate new values
 * <p>
 * 3. don't convert data to String (UTF-8), parse directly the bytes.
 * <p>
 * @author danesa
 *
 */
public abstract class ResourceCollector implements LinuxResourceCollector {

    protected RandomAccessFile procFile;
    protected byte[] buffer;
    protected static final int BUFFER_SIZE = 2048;
    protected NoProcResourceException invalidFileFormat = null;


    private String resourceName = null;
    // Possible (supported) params - to be initialized by classes extending ResourceCollector
    protected String[] params = null;
    // Possible params ids - an id for each of the possible parameters
    protected int[] ids = null;
    protected long[] values = null;

    public ResourceCollector(final String resourceName, final String fileName, final int nbParams) throws NoProcResourceException {
        this.resourceName = resourceName;
        try {
            procFile = new RandomAccessFile(fileName, "r");
            buffer = new byte[BUFFER_SIZE];
            int nbBytesRead = procFile.read(buffer);
            if (nbBytesRead == -1) {
                throw new NoProcResourceException("No data in " + procFile);
            }
        } catch (Exception e) {
            try {
                procFile.close();
            } catch (IOException ignore1) {
            }
            throw new NoProcResourceException(e);
        }
        invalidFileFormat = new NoProcResourceException("Invalid " + procFile + " file format");
        params = new String[nbParams];
        ids = new int[nbParams];
        for (int i = 0; i < ids.length; i++)
            ids[i] = i;
        values = new long[nbParams];
    }

    public int getParamId(String paramName) throws NoSuchLinuxResourceException {
        for (int i = 0; i < params.length; i++) {
            if (params[i].equalsIgnoreCase(paramName)) {
                return i;
            }
        }
        throw new NoSuchLinuxResourceException(paramName + " is not a supported parameter for resource " + resourceName);
    }

    public String getParamName(int paramId) throws NoSuchLinuxResourceException {
        try {
            return params[paramId];
        } catch (RuntimeException e) {
            throw new NoSuchLinuxResourceException(paramId + " does not correspond to any supported parameter for resource " + resourceName);
        }
    }

    public int[] getParamsIds() {
        return ids;
    }

    public String[] getParamsNames() {
        return params;
    }

    public long[] getValue(int[] paramIds) throws NoSuchLinuxResourceException,
    LinuxCollectingException {
        getUsage();

        long[] result = new long[paramIds.length];
        int i = 0;
        for (int id : paramIds) {
            result[i++] = values[id];
        }
        return result;
    }

    protected abstract void getUsage() throws LinuxCollectingException;
}
