/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.itests;

import java.io.FileReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.testng.Assert;
import org.testng.annotations.Test;

import au.com.bytecode.opencsv.CSVReader;

public class ChangeTest extends JOnASLauncher {

    final String name = "tst_change";
    final int period = 1;

    /**
     * Create an JMX indicator named 'tst_change':
     * - mbean: java.lang:type=Threading
     * - attributes: "ThreadCount,PeakThreadCount,TotalStartedThreadCount"
     * Check its creation.
     * @throws Exception
     */
    @Test
    public void testIndicatorCreate() throws Exception {
        logger.debug(">> testIndicatorCreate");
        // creation
        JasmineIndicator indicator = new JasmineIndicator();
        indicator.setName(name);
        indicator.setType("jmx");
        Map<String,String> props = new HashMap<String,String>();
        props.put("target", "agent0");
        props.put("mbean", "java.lang:type=Threading");
        props.put("attr", "ThreadCount,PeakThreadCount,TotalStartedThreadCount");
        indicator.setProperties(props);
        getProxyMXBean().createIndicator(indicator);
        // check
        JasmineIndicator ji = getProxyMXBean().getIndicator(name);
        Assert.assertEquals(ji.getName(), name , "Invalid indicator name. (Actual = " + ji.getName() + ", Expected = " + name + ")");
        Assert.assertEquals(ji.getType(), "jmx" , "Invalid indicator type. (Actual = " + ji.getType() + ", Expected = jmx)");
        Map<String, String> propsActual = ji.getProperties();
        Assert.assertEquals(propsActual.size(), props.size() , "Invalid indicator properties size. (Actual = " + props.size() + ", Expected = " + props.size() + ")");
    }

    /**
     * Create a file output named 'tst_change' with '/<tmp-dir>/tst_change.csv' as path
     * Check its creation.
     * @throws Exception
     */
    @Test
    public void testOutputFileCreate() throws Exception {
        logger.debug(">> testOutputFileCreate");
        // creation
        final String fileName = getTmpDirPath() + name + ".csv";
        JasmineOutput output = new JasmineOutput();
        output.setName(name);
        output.setType("file");
        Map<String,String> props = new HashMap<String,String>();
        props.put("path", fileName);
        output. setProperties(props);
        getProxyMXBean().createOutput(output);
        // check
        JasmineOutput jo = getProxyMXBean().getOutput(name);
        Assert.assertEquals(jo.getName(), name , "Invalid output name. (Actual = " + jo.getName() + ", Expected = " + name + ")");
        Assert.assertEquals(jo.getType(), "file" , "Invalid output type. (Actual = " + jo.getType() + ", Expected = file)");
        Map<String, String> propsActual = jo.getProperties();
        Assert.assertEquals(propsActual.size(), 1 , "Invalid output properties size. (Actual = " + propsActual.size() + ", Expected = 1)");
        Assert.assertEquals(propsActual.get("path"), fileName , "Invalid output properties values. (Actual = " + propsActual.get("path") + ", Expected = " + fileName + ")");
    }

    /**
     * Create a probe named 'tst_change':
     * - indicators: 'tst_change'
     * - output: 'tst_change'
     * - target: 'agent0' (jasmine-probe server itself)
     * Check its creation.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testIndicatorCreate","testOutputFileCreate"})
    public void testProbeCreate() throws Exception {
        logger.debug(">> testProbeCreate");
        // creation
        JasmineProbe probe = new JasmineProbe();
        probe.setId(name);
        List<String> ji = new ArrayList<String>(Arrays.asList(name));
        probe.setIndicatorList(ji);
        List<String> jo = new ArrayList<String>(Arrays.asList(name));
        probe.setOutputList(jo);
        List<String> jt = new ArrayList<String>(Arrays.asList("agent0"));
        probe.setTargetList(jt);
        probe.setPeriod(period);
        getProxyMXBean().createProbe(probe);
        // check
        JasmineProbe jp = getProxyMXBean().getProbe(name);
        Assert.assertEquals(jp.getId(), name , "Invalid probe id. (Actual = " + jp.getId() + ", Expected = " + name + ")");
        Assert.assertEquals(jp.getId(), name , "Invalid probe id. (Actual = " + jp.getId() + ", Expected = " + name + ")");
        Assert.assertEquals(jp.getPeriod(), period , "Invalid probe period. (Actual = " + jp.getPeriod() + ", Expected = " + period + ")");
        Assert.assertEquals(jp.getStatus(), JasmineProbe.PROBE_STOPPED , "Invalid probe status. (Actual = " + jp.getStatus() + ", Expected = " + JasmineProbe.PROBE_STOPPED + ")");
    }

    /**
     * Start the probe named 'tst_change'.
     * Wait a while some collects.
     * Check the validity of the collects.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeCreate"})
    public void testProbeStart() throws Exception {
        logger.debug(">> testProbeStart");
        final String fileName = getTmpDirPath() + name + ".csv";
        // start the probe
        getProxyMXBean().startProbe(name);
        // Check the probe status
        JasmineProbe jp = getProxyMXBean().getProbe(name);
        if ((jp.getStatus() != JasmineProbe.PROBE_STARTED) && (jp.getStatus() != JasmineProbe.PROBE_RUNNING)) {
            Assert.fail("Invalid probe status (Actual = " + jp.printStatus() + ", Expected = STARTED|RUNNING)");
        }
        // Wait some collects ...
        Thread.sleep(period*1000*2);
        // stop the probe
        getProxyMXBean().stopProbe(name);
        // Check the probe status
        jp = getProxyMXBean().getProbe(name);
        Assert.assertEquals(jp.getStatus(), JasmineProbe.PROBE_STOPPED , "Invalid probe status. (Actual = " + jp.printStatus() + ", Expected = STOPPED)");
        // Check the output file content
        // - not empty
        // - check the indicator name
        CSVReader reader = null;
        boolean emptyOutput = true;
        String indicatorName;
        try {
            reader = new CSVReader(new FileReader(fileName), ';');
            String[] nextLine;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                if ( ! "tst_change.ThreadCount".equals(indicatorName) && ! "tst_change.PeakThreadCount".equals(indicatorName) && ! "tst_change.TotalStartedThreadCount".equals(indicatorName)) {
                    Assert.fail("Invalid indicator name: " + indicatorName);
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe " + name);
    }

    /**
     * Change the 'tst_change' indicator used by the 'tst_change' probe when the probe is currently running.
     * - Take off the 'TotalStartedThreadCount' attribute and only keep 'ThreadCount' and 'PeakThreadCount' attributes.
     * Wait a while some collects
     * Stop the 'tst_change' probe.
     * Check the change is taken into account : the 'TotalStartedThreadCount' attribute must no more be collected.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeStart"})
    public void testInjectorChange() throws Exception {
        logger.debug(">> testInjectorChange");
        final String fileName = getTmpDirPath() + name + ".csv";
        // restart the probe
        getProxyMXBean().startProbe(name);
        // change the indicator
        JasmineIndicator ji = getProxyMXBean().getIndicator(name);
        Map<String,String> props = new HashMap<String,String>();
        props.put("target", "agent0");
        props.put("mbean", "java.lang:type=Threading");
        props.put("attr", "ThreadCount,PeakThreadCount");
        ji.setProperties(props);
        long timeChange = System.currentTimeMillis();
        getProxyMXBean().changeIndicator(ji);
        // Wait some collects ...
        Thread.sleep(period*1000*3);
        // stop the probe
        getProxyMXBean().stopProbe(name);
        // Check the output file content
        // - there is no more TotalStartedThreadCount since the timestamp timeChange
        CSVReader reader = null;
        boolean emptyOutput = true;
        String indicatorName;
        long time;
        try {
            reader = new CSVReader(new FileReader(fileName), ';');
            String[] nextLine;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                time = Long.parseLong(nextLine[Util.CSV_INDEX_TIME]);
                if ( ! "tst_change.ThreadCount".equals(indicatorName) && ! "tst_change.PeakThreadCount".equals(indicatorName) && ! "tst_change.TotalStartedThreadCount".equals(indicatorName)) {
                    Assert.fail("Invalid indicator name: " + indicatorName);
                }
                if ("tst_change.TotalStartedThreadCount".equals(indicatorName) && (time > timeChange)) {
                    Assert.fail("The indicator tst_change.TotalStartedThreadCount must no more collected since the time " + timeChange);
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe " + name);
    }
    
    /**
     * Change the path of the 'tst_change' output file used by the 'tst_change' probe when the probe is currently running:
     * - old path: '/<tmp-dir>/tst_change.csv'
     * - new path: '/<tmp-dir>/tst_change_2.csv'
     * Wait a while some collects
     * Stop the 'tst_change' probe.
     * Check the change is taken into account : collects must no more be stored in tst_change.csv file but in the tst_change_2.csv file.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testInjectorChange"})
    public void testOutputChange() throws Exception {
        logger.debug(">> testOutputChange");
        final String fileName1 = getTmpDirPath() + name + ".csv";
        final String fileName2 = getTmpDirPath() + name + "_2.csv";
        // restart the probe
        getProxyMXBean().startProbe(name);
        // change the indicator
        JasmineOutput jo = getProxyMXBean().getOutput(name);
        Map<String,String> props = new HashMap<String,String>();
        props.put("path", fileName2);
        jo.setProperties(props);
        long timeChange = System.currentTimeMillis();
        getProxyMXBean().changeOutput(jo);
        // Wait some collects ...
        Thread.sleep(period*1000*3);
        // stop the probe
        getProxyMXBean().stopProbe(name);
        // Check the first output file content
        // - there is no more collect since the timestamp timeChange in the first file
        CSVReader reader = null;
        boolean emptyOutput = true;
        long time;
        try {
            reader = new CSVReader(new FileReader(fileName1), ';');
            String[] nextLine;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                time = Long.parseLong(nextLine[Util.CSV_INDEX_TIME]);
                if ((time > timeChange)) {
                    Assert.fail("There must be no more collected since the time " + timeChange + " in the file " + fileName1);
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe " + name + " in the first output");
        // Check the second output file content
        // - there is collect since the timestamp timeChange in the second file
        reader = null;
        emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fileName2), ';');
            String[] nextLine;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                time = Long.parseLong(nextLine[Util.CSV_INDEX_TIME]);
                if ((time < timeChange)) {
                    Assert.fail("There must be no collect before the time " + timeChange + " in the file " + fileName2);
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe " + name + " in the second output");
    }
}
