/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
 
package org.ow2.jasmine.probe.itests;


import java.io.FileReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;

import au.com.bytecode.opencsv.CSVReader;

import org.testng.Assert;
import org.testng.annotations.Test;

public class IndicatorDfTest extends JOnASLauncher {

    final String name = "tst_DfAll";

    /**
     * Create an DF indicator named 'tst_DfAll' without 'disk' property defined.
     * Check its creation.
     * @throws Exception
     */
    @Test
    public void testIndicatorCreateDfAll() throws Exception {
        logger.debug(">> testIndicatorCreateDfAll");
        // creation
        JasmineIndicator indicatorDfAll = new JasmineIndicator();
        indicatorDfAll.setName(name);
        indicatorDfAll.setType("df");
        getProxyMXBean().createIndicator(indicatorDfAll);
        // check
        JasmineIndicator ji = getProxyMXBean().getIndicator(name);
        Assert.assertEquals(ji.getName(), name , "Invalid indicator name. (Actual = " + ji.getName() + ", Expected = " + name + ")");
        Assert.assertEquals(ji.getType(), "df" , "Invalid indicator type. (Actual = " + ji.getType() + ", Expected = df)");
        Map<String, String> props = ji.getProperties();
        Assert.assertEquals(props.size(), 0 , "Invalid indicator properties size. (Actual = " + props.size() + ", Expected = 0)");
    }

    /**
     * Create a file output named 'tst_DfAll' with '/<tmp-dir>/tst_DfAll.csv' as path
     * Check its creation.
     * @throws Exception
     */
    @Test
    public void testOutputFileCreateDfAll() throws Exception {
        logger.debug(">> testOutputFileDfAll");
        // creation
        final String fileName = getTmpDirPath() + name + ".csv";
        JasmineOutput outputDfAll = new JasmineOutput();
        outputDfAll.setName(name);
        outputDfAll.setType("file");
        Map<String,String> props = new HashMap<String,String>();
        props.put("path", fileName);
        outputDfAll. setProperties(props);
        getProxyMXBean().createOutput(outputDfAll);
        // check
        JasmineOutput jo = getProxyMXBean().getOutput(name);
        Assert.assertEquals(jo.getName(), name , "Invalid output name. (Actual = " + jo.getName() + ", Expected = " + name + ")");
        Assert.assertEquals(jo.getType(), "file" , "Invalid output type. (Actual = " + jo.getType() + ", Expected = file)");
        Map<String, String> propsActual = jo.getProperties();
        Assert.assertEquals(propsActual.size(), 1 , "Invalid output properties size. (Actual = " + propsActual.size() + ", Expected = 1)");
        Assert.assertEquals(propsActual.get("path"), fileName , "Invalid output properties values. (Actual = " + propsActual.get("path") + ", Expected = " + fileName + ")");
    }

    /**
     * Create a probe named 'tst_DfAll':
     * - indicators: 'tst_DfAll'
     * - output: 'tst_DfAll'
     * - target: 'agent0' (jasmine-probe server itself)
     * Check its creation.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testIndicatorCreateDfAll","testOutputFileCreateDfAll"})
    public void testProbeCreateDfAll() throws Exception {
        logger.debug(">> testProbeCreateDfAll");
        // creation
        final int period = 1;
        JasmineProbe probeDfAll = new JasmineProbe();
        probeDfAll.setId(name);
        List<String> ji = new ArrayList<String>(Arrays.asList(name));
        probeDfAll.setIndicatorList(ji);
        List<String> jo = new ArrayList<String>(Arrays.asList(name));
        probeDfAll.setOutputList(jo);
        List<String> jt = new ArrayList<String>(Arrays.asList("agent0"));
        probeDfAll.setTargetList(jt);
        probeDfAll.setPeriod(period);
        getProxyMXBean().createProbe(probeDfAll);
        // check
        JasmineProbe jp = getProxyMXBean().getProbe(name);
        Assert.assertEquals(jp.getId(), name , "Invalid probe id. (Actual = " + jp.getId() + ", Expected = " + name + ")");
        Assert.assertEquals(jp.getId(), name , "Invalid probe id. (Actual = " + jp.getId() + ", Expected = " + name + ")");
        Assert.assertEquals(jp.getPeriod(), period , "Invalid probe period. (Actual = " + jp.getPeriod() + ", Expected = " + period + ")");
        Assert.assertEquals(jp.getStatus(), JasmineProbe.PROBE_STOPPED , "Invalid probe status. (Actual = " + jp.getStatus() + ", Expected = " + JasmineProbe.PROBE_STOPPED + ")");
    }

    /**
     * Start the probe named 'tst_DfAll'.
     * Wait a while some collects.
     * Stop the probe.
     * Check the validity of the collects.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeCreateDfAll"})
    public void testProbeStartStopDfAll() throws Exception {
        logger.debug(">> testProbeStartStopDfAll");
        // start the probe DfAll
        final int period = 1;
        final String fileName = getTmpDirPath() + "tst_DfAll.csv";
        getProxyMXBean().startProbe(name);
        // Check the probe status
        JasmineProbe jp = getProxyMXBean().getProbe(name);
        if ((jp.getStatus() != JasmineProbe.PROBE_STARTED) && (jp.getStatus() != JasmineProbe.PROBE_RUNNING)) {
            Assert.fail("Invalid probe status (Actual = " + jp.printStatus() + ", Expected = STARTED|RUNNING)");
        }
        // Wait some collects ...
        Thread.sleep(period*1000*2);
        // stop the probe
        getProxyMXBean().stopProbe(name);
        // Check the probe status
        jp = getProxyMXBean().getProbe(name);
        Assert.assertEquals(jp.getStatus(), JasmineProbe.PROBE_STOPPED , "Invalid probe status. (Actual = " + jp.printStatus() + ", Expected = STOPPED)");
        // Check the output file content
        // - not empty
        // - the collected value is a pourcentage and so must be beetween 0 and 100
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fileName), ';');
            String[] nextLine;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                int diskUsed = Integer.parseInt(nextLine[Util.CSV_INDEX_VALUE]);
                if ((diskUsed < 0) || (diskUsed > 100)) {
                    Assert.fail("The 'disk used' collected value for Df is NOT a pourcentage value (" + diskUsed + ")");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe " + name);
    }

    /**
     * Remove the probe named 'tst_DfAll'.
     * Check its destruction.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeStartStopDfAll"})
    public void testProbeRemoveDfAll() throws Exception {
        logger.debug(">> testProbeRemoveDfAll");
        // remove
        getProxyMXBean().removeProbe(name);
        // check
        boolean ko = true;
        try {
            getProxyMXBean().getProbe(name);
        } catch (Exception e) {
            ko = false;
        }
        Assert.assertFalse(ko, "The probe '" + name + "' should no more exist");
        
    }

    /**
     * Remove the output named 'tst_DfAll'.
     * Check its destruction.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeRemoveDfAll"})
    public void testOuputFileRemoveDfAll() throws Exception {
        logger.debug(">> testOuputFileRemoveDfAll");
        // remove
        getProxyMXBean().removeOutput(name);
        // check
        boolean ko = true;
        try {
            getProxyMXBean().getOutput(name);
        } catch (Exception e) {
            ko = false;
        }
        Assert.assertFalse(ko, "The output file '" + name + "' should no more exist");
        
    }

    /**
     * Remove the indicator named 'tst_DfAll'.
     * Check its destruction.
     * @throws Exception
     */
    @Test(dependsOnMethods={"testProbeRemoveDfAll"})
    public void testIndicatorRemoveDfAll() throws Exception {
        logger.debug(">> testIndicatorRemoveDfAll");
        // remove
        getProxyMXBean().removeIndicator(name);
        // check
        boolean ko = true;
        try {
            getProxyMXBean().getIndicator(name);
        } catch (Exception e) {
            ko = false;
        }
        Assert.assertFalse(ko, "The indicator '" + name + "' should no more exist");
        
    }
}
