/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.itests;

import java.io.FileReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.itests.dummymbeanservice.Basic;
import org.testng.Assert;
import org.testng.annotations.Test;

import au.com.bytecode.opencsv.CSVReader;

public abstract class AbsProcessingIndicatorTest extends JOnASLauncher {

    /*
     * The waiting time to get time to do some collects
     */
    static final int WAIT_COLLECTS_IN_SEC = 70;

    /*
     * Accepted delta value for a actual VPeriod to a expected VPeriod
     */
    private static final int VPERIOD_ACCEPTED_DELTA = 3;

    // Theorical Period (See probe definitions)
    private static final int THEO_PERIOD = 3;


    private static final Double PRECISION_DOUBLE = 0.001;


    /**
     * Check the collects for the 'mbdefault' mbean corresponding to the ptst_mbdefault probe.
     */
    @Test
    public void testMbdefaultAllAttributes() throws Exception {
        logger.debug(">> testMbdefaultAllAttributes");
        String fName = getTmpDirPath() + "tst_mbdefault_all_attributes.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            boolean bFirstVPeriod = true;
            int minVPeriod = 100;
            int maxVPeriod = 0;
            int nbVPeriod = 0;
            int deltasVPeriod = 0;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mbdefault_all_attributes.Name".equals(indicatorName)) {
                    Assert.assertEquals(value, "mbdefault", "Invalid 'Name' value !");
                } else if ("tst_mbdefault_all_attributes.CAmplitude".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_AMPLITUDE, "Invalid 'CAmplitude' value !");
                } else if ("tst_mbdefault_all_attributes.CIncrement".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_INCREMENT, "Invalid 'CIncrement' value !");
                } else if ("tst_mbdefault_all_attributes.CStep".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_STEP, "Invalid 'CStep' value !");
                } else if ("tst_mbdefault_all_attributes.VIncremented".equals(indicatorName)) {
                    // do nothing
                } else if ("tst_mbdefault_all_attributes.VDescription".equals(indicatorName)) {
                    // do nothing
                } else if ("tst_mbdefault_all_attributes.VPeriod".equals(indicatorName)) {
                    if (bFirstVPeriod) {
                        // Ignore the first VPeriod collect which is not a period between two collects.
                        bFirstVPeriod = false;
                    } else {
                        // The VPeriod is not exactly the period value of the probe.
                        // Get VPeriod information to do some later check
                        int vp = Integer.parseInt(value);
                        if (minVPeriod > vp) {
                            minVPeriod = vp;
                        }
                        if (maxVPeriod < vp) {
                            maxVPeriod = vp;
                        }
                        nbVPeriod++;
                        deltasVPeriod = deltasVPeriod + Math.abs(vp - THEO_PERIOD);
                    }
                } else if ("tst_mbdefault_all_attributes.VSinusoidal".equals(indicatorName)) {
                    Assert.assertTrue(between(value, -Basic.DEFAULT_AMPLITUDE, Basic.DEFAULT_AMPLITUDE), "Invalid 'VSinusoidal' value ! (" + value + ")");
                } else if ("tst_mbdefault_all_attributes.CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_INT_1, "Invalid 'CInt1' value !");
                } else if ("tst_mbdefault_all_attributes.CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_INT_2, "Invalid 'CInt2' value !");
                } else if ("tst_mbdefault_all_attributes.CFloat1".equals(indicatorName)) {
                    Assert.assertEquals(Float.parseFloat(value), Basic.DEFAULT_FLOAT_1, PRECISION_DOUBLE, "Invalid 'CFloat1' value !");
                } else if ("tst_mbdefault_all_attributes.CFloat2".equals(indicatorName)) {
                    Assert.assertEquals(Float.parseFloat(value), Basic.DEFAULT_FLOAT_2, PRECISION_DOUBLE, "Invalid 'CFloat2' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
            logger.debug(">>>> period-min = " + minVPeriod + ", period-max = " + maxVPeriod + " , period-nb = " + nbVPeriod + ", period-deltas = " + deltasVPeriod);
            if (minVPeriod < THEO_PERIOD - VPERIOD_ACCEPTED_DELTA) {
                Assert.fail("Invalid VPeriod value " + minVPeriod);
            }
            if (maxVPeriod > THEO_PERIOD + VPERIOD_ACCEPTED_DELTA) {
                Assert.fail("Invalid VPeriod value " + maxVPeriod);
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe 'ptst_mbdefault' in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_01' mbean corresponding to the ptst_mb_aa_01_delta probe.
     * Collects a derived DELTA value
     */
    @Test
    public void testMb_aa_01_DerivedDelta() throws Exception {
        logger.debug(">> testMb_aa_01_DerivedDelta");
        String fName = getTmpDirPath() + "tst_mb_aa_01_delta.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_01_CIncrement".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1, "Invalid 'CIncrement' value !");
                } else if ("tst_mb_aa_01_delta_VIncremented".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 1, "Invalid  value for 'delta_VIncremented' !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe 'ptst_mb_aa_01_delta' in the output " + fName);
    }

    /**
     * Test to reproduce the bug PROBE-148.
     * Add the tst_mb_aa_01_VIncremented indicator in the ptst_mb_aa_01_delta probe definition and check that all the collects are correctly done.
     */
    @Test(dependsOnMethods={"testMb_aa_01_DerivedDelta"})
    public void testMb_aa_01_AddIndicator() throws Exception {
        logger.debug(">> testMb_aa_01_AddIndicator");
        final String fName = getTmpDirPath() + "tst_mb_aa_01_add_indicator.csv";
        // restart the probe
        getProxyMXBean().startProbe("ptst_mb_aa_01_delta");
        // Create the new output file
        JasmineOutput output = new JasmineOutput();
        output.setName(fName);
        output.setType("file");
        Map<String,String> props = new HashMap<String,String>();
        props.put("path", fName);
        output. setProperties(props);
        getProxyMXBean().createOutput(output);
        // Add the tst_mb_aa_01_VIncremented indicator in the ptst_mb_aa_01_delta probe definition
        JasmineProbe jp = getProxyMXBean().getProbe("ptst_mb_aa_01_delta");
        List<String> il = jp.getIndicatorList();
        il.add("tst_mb_aa_01_VIncremented");
        jp.setIndicatorList(il);
        List<String> ol = new ArrayList<String>();
        ol.add(fName);
        jp.setOutputList(ol);
        jp.setPeriod(2);
        getProxyMXBean().changeProbe(jp);
        // Wait some time to do some collects, and than stop the probe
        Thread.sleep(1000*10);
        getProxyMXBean().stopProbe("ptst_mb_aa_01_delta");
        // Check the collects
        CSVReader reader = null;
        int nbCI = 0;
        int nbVI = 0;
        int nbDeltaVI = 0;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_01_CIncrement".equals(indicatorName)) {
                    nbCI++;
                    Assert.assertEquals(Integer.parseInt(value), 1, "Invalid 'CIncrement' value !");
                } else if ("tst_mb_aa_01_VIncremented".equals(indicatorName)) {
                    nbVI++;
                } else if ("tst_mb_aa_01_delta_VIncremented".equals(indicatorName)) {
                    nbDeltaVI++;
                    if (nbDeltaVI > 1) {
                        // Do not check the first value because may be randomly different
                        Assert.assertEquals(Integer.parseInt(value), 1, "Invalid  value for 'delta_VIncremented' !");
                    }
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(nbCI==0, "There is no collects for the probe 'ptst_mb_aa_01_delta' in the output " + fName);
        Assert.assertEquals(nbVI, nbCI, "There is not enough collected values for indicator tst_mb_aa_01_VIncremented for the probe 'ptst_mb_aa_01_delta' in the output " + fName);
        // TODO This test does not pass randomly with the error:
        //      'There is not enough collected values for indicator tst_mb_aa_01_delta_VIncremented'
        // So, just test there is some collected values for this indicator
        //Assert.assertEquals(nbDeltaVI, nbCI-1, "There is not enough collected values for indicator tst_mb_aa_01_delta_VIncremented for the probe 'ptst_mb_aa_01_delta' in the output " + fName);
        Assert.assertTrue(nbDeltaVI > 0, "There is no collected values for indicator tst_mb_aa_01_delta_VIncremented for the probe 'ptst_mb_aa_01_delta' in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_02' mbean corresponding to the ptst_mb_aa_02 probe.
     * Collects some attributes and a derived RATE value
     */
    @Test
    public void testMb_aa_02_DerivedRate() throws Exception {
        logger.debug(">> testMb_aa_02_DerivedRate");
        String fName = getTmpDirPath() + "tst_mb_aa_02_rate.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_02_CIncrement".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1, "Invalid 'CIncrement' value !");
                } else if ("tst_mb_aa_02_rate_VIncremented".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertTrue(between(value, (1.0/(THEO_PERIOD + VPERIOD_ACCEPTED_DELTA)), (1.0/(THEO_PERIOD - VPERIOD_ACCEPTED_DELTA))), "Invalid 'rate_VIncremented' value " + value);
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe 'ptst_mb_aa_02_rate' in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_03' mbean corresponding to the ptst_mb_aa_03 probe.
     * Collects some attributes and a SLOPE value
     */
    @Test
    public void testMb_aa_03_Slope() throws Exception {
        logger.debug(">> testMb_aa_03_Slope");
        String fName = getTmpDirPath() + "tst_mb_aa_03_slope.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_03.CIncrement".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1, "Invalid 'CIncrement' value !");
                } else if ("tst_mb_aa_03.CStep".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 2, "Invalid 'CStep' value !");
                } else if ("tst_mb_aa_03_slope_VIncremented_VSinusoidal".equals(indicatorName)) {
                    emptyOutput = false;
                    Double dAbsValue = StrictMath.abs(Double.parseDouble(value));
                    Assert.assertEquals(dAbsValue, 0.5, PRECISION_DOUBLE, "Invalid 'slope_VIncremented_VSinusoidal' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects for the probe 'ptst_mb_aa_03_slope' in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_add probe.
     * Collects some attributes and a correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelateAdd() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateAdd");
        String fName = getTmpDirPath() + "tst_mb_aa_04_add.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'CInt2' value !");
                } else if ("tst_mb_aa_04_add_CInt1_CInt2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 1750, "Invalid 'add_CInt1_Int2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_sub probe.
     * Collects some attributes and a correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelateSub() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateSub");
        String fName = getTmpDirPath() + "tst_mb_aa_04_sub.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'CInt2' value !");
                } else if ("tst_mb_aa_04_sub_CInt1_CInt2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 1250, "Invalid 'sub_CInt1_Int2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_mul probe.
     * Collects some attributes and a correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelateMul() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateMul");
        String fName = getTmpDirPath() + "tst_mb_aa_04_mul.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'CInt2' value !");
                } else if ("tst_mb_aa_04_mul_CInt1_CInt2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 375000, "Invalid 'mul_CInt1_Int2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_div probe.
     * Collects some attributes and a correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelateDiv() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateDiv");
        String fName = getTmpDirPath() + "tst_mb_aa_04_div.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'CInt2' value !");
                } else if ("tst_mb_aa_04_div_CInt1_CInt2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Double.parseDouble(value), 6, PRECISION_DOUBLE, "Invalid 'div_CInt1_Int2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_percent probe.
     * Collects some attributes and a correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelatePercent() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelatePercent");
        String fName = getTmpDirPath() + "tst_mb_aa_04_percent.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'CInt2' value !");
                } else if ("tst_mb_aa_04_percent_CInt1_CInt2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Double.parseDouble(value), 600, PRECISION_DOUBLE, "Invalid 'percent_CInt1_Int2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_addComplex probe.
     * Collects a complex correlate ADD value
     */
    @Test
    public void testMb_aa_04_CorrelateAddComplex() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateAddComplex");
        String fName = getTmpDirPath() + "tst_mb_aa_04_add_complex.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'tst_mb_aa_04_CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'tst_mb_aa_04_CInt2' value !");
                } else if ("tst_mb_aa_04_add_CInt1_CInt2_CInt1_15".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 3265, "Invalid 'tst_mb_aa_04_add_CInt1_CInt2_CInt1_15' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_mulComplex probe.
     * Collects a complex correlate MUL value
     */

    @Test
    public void testMb_aa_04_CorrelateMulComplex() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateMulComplex");
        String fName = getTmpDirPath() + "tst_mb_aa_04_mul_complex.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 1500, "Invalid 'tst_mb_aa_04_CInt1' value !");
                } else if ("tst_mb_aa_04_CInt2".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), 250, "Invalid 'tst_mb_aa_04_CInt2' value !");
                } else if ("tst_mb_aa_04_mul_CInt1_CInt2_CInt1_2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Integer.parseInt(value), 1125000000, "Invalid 'tst_mb_aa_04_mul_CInt1_CInt2_CInt1_2' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_ab_xx' mbean corresponding to the ptst_mb_ab_xx_XXXX probes.
     * Collects correlate ADD , SUB, MUL, DIV, PERCENT, values
     */
    @Test
    public void testMb_ab_XX_Correlate() throws Exception {
        logger.debug(">> testMb_ab_XX_Correlate");
        String fName = getTmpDirPath() + "tst_mb_ab_xx_correlate.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_ab_xx_add_VIncremented".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 0, "Invalid 'tst_mb_ab_xx_add_VIncremented' value !");
                } else if ("tst_mb_ab_01_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 500, "Invalid 'tst_mb_ab_01_CInt1' value !");
                } else if ("tst_mb_ab_02_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 100, "Invalid 'tst_mb_ab_02_CInt1' value !");
                } else if ("tst_mb_ab_xx_sub_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 400, "Invalid 'tst_mb_ab_xx_sub_CInt1' value !");
                } else if ("tst_mb_ab_xx_mul_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 50000, "Invalid 'tst_mb_ab_xx_mul_CInt1' value !");
                } else if ("tst_mb_ab_xx_div_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 5, PRECISION_DOUBLE, "Invalid 'tst_mb_ab_xx_div_CInt1' value !");
                } else if ("tst_mb_ab_xx_percent_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 500, PRECISION_DOUBLE, "Invalid 'tst_mb_ab_xx_percent_CInt1' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Collects correlate SUM, AVERAGE, MIN , MAX, of several attributes of one mbean
     */
    @Test
    public void testMb_ac_01_Merge() throws Exception {
        logger.debug(">> testMb_ac_01_Merge");
        String fName = getTmpDirPath() + "tst_mb_ac_01_merge.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_ac_01.CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 500, "Invalid 'tst_mb_ab_01.CInt1' value !");
                } else if ("tst_mb_ac_01.CInt2".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), -100, "Invalid 'tst_mb_ab_01.CInt2' value !");
                } else if ("tst_mb_ac_01_min".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), -100, "Invalid 'tst_mb_ab_01_min' value !");
                } else if ("tst_mb_ac_01_max".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 500, "Invalid 'tst_mb_ab_01_max' value !");
                } else if ("tst_mb_ac_01_sum".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 400, "Invalid 'tst_mb_ab_01_sum' value !");
                } else if ("tst_mb_ac_01_average".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 200, PRECISION_DOUBLE, "Invalid 'tst_mb_ab_01_average' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Collects correlate SUM, AVERAGE, MIN , MAX, of one attribute of severals mbeans
     */
    @Test
    public void testMb_ac_xx_Merge() throws Exception {
        logger.debug(">> testMb_ac_xx_Merge");
        String fName = getTmpDirPath() + "tst_mb_ac_xx_merge_one.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_ac_xx_CInt1".equals(indicatorName)) {
                } else if ("tst_mb_ac_xx_sum_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 800, "Invalid 'tst_mb_ac_xx_sum_CInt1' value !");
                } else if ("tst_mb_ac_xx_average_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 400.0, PRECISION_DOUBLE, "Invalid 'tst_mb_ac_xx_average_CInt1' value !");
                } else if ("tst_mb_ac_xx_min_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 300, "Invalid 'tst_mb_ac_xx_min_CInt1' value !");
                } else if ("tst_mb_ac_xx_max_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 500, "Invalid 'tst_mb_ac_xx_max_CInt1' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the target define at the probe level is taken into account.
     */
    @Test
    public void testMbdefaultTargetOnProbe() throws Exception {
        logger.debug(">> testMbdefaultTargetOnProbe");
        String fName = getTmpDirPath() + "tst_mbdefault_target_on_probe.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if  ("tst_mbdefault_target_unknown.CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), Basic.DEFAULT_INT_1, "Invalid 'CInt1' value !");
                } else if ("tst_mbdefault_target_unknown.CFloat1".equals(indicatorName)) {
                    Assert.assertEquals(Float.parseFloat(value), Basic.DEFAULT_FLOAT_1, PRECISION_DOUBLE, "Invalid 'CFloat1' value !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_add_float probe.
     * Collects some attributes and a correlate ADD value on float sources
     */
    @Test
    public void testMb_aa_04_CorrelateAddFloat() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateAddFloat");
        String fName = getTmpDirPath() + "tst_mb_aa_04_add_float.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CFloat1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 150.4, PRECISION_DOUBLE, "Invalid 'CFloat1' value !");
                } else if ("tst_mb_aa_04_CFloat2".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 2.0, PRECISION_DOUBLE, "Invalid 'CFloat2' value !");
                } else if ("tst_mb_aa_04_add_CFloat1_CFloat2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Double.parseDouble(value), 152.4, PRECISION_DOUBLE, "Invalid 'add_CFloat1_CFloat2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_sub_float probe.
     * Collects some attributes and a correlate SUB value on float sources
     */
    @Test
    public void testMb_aa_04_CorrelateSubFloat() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateSubFloat");
        String fName = getTmpDirPath() + "tst_mb_aa_04_sub_float.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CFloat1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 150.4, PRECISION_DOUBLE, "Invalid 'CFloat1' value !");
                } else if ("tst_mb_aa_04_CFloat2".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 2.0, PRECISION_DOUBLE, "Invalid 'CFloat2' value !");
                } else if ("tst_mb_aa_04_sub_CFloat1_CFloat2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Double.parseDouble(value), 148.4, PRECISION_DOUBLE, "Invalid 'sub_CFloat1_CFloat2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_04' mbean corresponding to the ptst_mb_aa_04_mul_float probe.
     * Collects some attributes and a correlate MUL value on float sources
     */
    @Test
    public void testMb_aa_04_CorrelateMulFloat() throws Exception {
        logger.debug(">> testMb_aa_04_CorrelateMulFloat");
        String fName = getTmpDirPath() + "tst_mb_aa_04_mul_float.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_04_CFloat1".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 150.4, PRECISION_DOUBLE, "Invalid 'CFloat1' value !");
                } else if ("tst_mb_aa_04_CFloat2".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 2.0, PRECISION_DOUBLE, "Invalid 'CFloat2' value !");
                } else if ("tst_mb_aa_04_mul_CFloat1_CFloat2".equals(indicatorName)) {
                    emptyOutput = false;
                    Assert.assertEquals(Double.parseDouble(value), 300.8, PRECISION_DOUBLE, "Invalid 'mul_CFloat1_CFloat2' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_ac_xx' mbeans corresponding to the ptst_mb_ac_xx_merge_float probe.
     * Merge SUM, AVERAGE, MIN, MAX values on severals FLOAT attributes of severals mbeans
     */
    @Test
    public void testMb_ac_xx_MergeFloat() throws Exception {
        logger.debug(">> testMb_ac_xx_MergeFloat");
        String fName = getTmpDirPath() + "tst_mb_ac_xx_merge_float.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_ac_xx_CFloatX.CFloat1".equals(indicatorName)) {
                } else if ("tst_mb_ac_xx_CFloatX.CFloat2".equals(indicatorName)) {
                } else if ("tst_mb_ac_xx_sum_CFloatX".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 760.8, PRECISION_DOUBLE, "Invalid 'tst_mb_ac_xx_sum_CFloatX' value !");
                } else if ("tst_mb_ac_xx_average_CFloatX".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 190.2, PRECISION_DOUBLE, "Invalid 'tst_mb_ac_xx_average_CFloatX' value ");
                } else if ("tst_mb_ac_xx_max_CFloatX".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), 500.5, PRECISION_DOUBLE, "Invalid 'tst_mb_ac_xx_max_CFloatX' value ");
                } else if ("tst_mb_ac_xx_min_CFloatX".equals(indicatorName)) {
                    Assert.assertEquals(Double.parseDouble(value), -100.5, PRECISION_DOUBLE, "Invalid 'tst_mb_ac_xx_min_CFloatX' value ");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Test to reproduce the bug PROBE-133.
     * Check the collects for the 'mb_aa_05' mbean corresponding to the ptst_mb_aa_05_BUG133_several_use_indicator probe.
     */
    @Test
    public void testMb_aa_05_Bug133() throws Exception {
        logger.debug(">> testMb_aa_05_Bug133");
        String fName = getTmpDirPath() + "tst_mb_aa_05_BUG133_several_use_indicator.csv";
        CSVReader reader = null;
        boolean emptyOutputAdd = true;
        boolean emptyOutputSub = true;
        boolean emptyOutputDelta = true;
        boolean emptyOutputRate = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_05_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 15, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_05_CInt2".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 25, "Invalid  value for 'tst_mb_aa_05_CInt2' !");
                } else if ("tst_mb_aa_05_add_CInt1_1".equals(indicatorName)) {
                    emptyOutputAdd = false;
                    Assert.assertEquals(Integer.parseInt(value), 16, "Invalid  value for 'tst_mb_aa_05_add_CInt1_1' !");
                } else if ("tst_mb_aa_05_sub_CInt1_1".equals(indicatorName)) {
                    emptyOutputSub = false;
                    Assert.assertEquals(Integer.parseInt(value), 14, "Invalid  value for 'tst_mb_aa_05_sub_CInt1_1' !");
                } else if ("tst_mb_aa_05_delta_CInt1".equals(indicatorName)) {
                    emptyOutputDelta = false;
                    Assert.assertEquals(Integer.parseInt(value), 0, "Invalid  value for 'tst_mb_aa_05_delta_CInt1' !");
                } else if ("tst_mb_aa_05_rate_CInt1".equals(indicatorName)) {
                    emptyOutputRate = false;
                    Assert.assertEquals(Double.parseDouble(value), 0.0, "Invalid  value for 'tst_mb_aa_05_rate_CInt1' !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutputAdd, "There is no collects for the indicator 'tst_mb_aa_05_add_CInt1_1' in the output " + fName);
        Assert.assertFalse(emptyOutputSub, "There is no collects for the indicator 'tst_mb_aa_05_sub_CInt1_1' in the output " + fName);
        Assert.assertFalse(emptyOutputDelta, "There is no collects for the indicator 'tst_mb_aa_05_delta_CInt1' in the output " + fName);
        Assert.assertFalse(emptyOutputRate, "There is no collects for the indicator 'tst_mb_aa_05_rate_CInt1' in the output " + fName);
    }

    /**
     * Test to reproduce the bug PROBE-148.
     * Remove the tst_mb_aa_05_rate_CInt1 indicator to the ptst_mb_aa_05_BUG133_several_use_indicator probe definition and check the collects are correct.
     */
    @Test(dependsOnMethods="testMb_aa_05_Bug133")
    public void testMb_aa_05_RemoveIndicator() throws Exception {
        logger.debug(">> testMb_aa_05_RemoveIndicator");
        final String fName = getTmpDirPath() + "tst_mb_aa_05_remove_indicator.csv";
        // Restart the probe
        getProxyMXBean().startProbe("ptst_mb_aa_05_BUG133_several_use_indicator");
        // Create the new output file
        JasmineOutput output = new JasmineOutput();
        output.setName(fName);
        output.setType("file");
        Map<String,String> props = new HashMap<String,String>();
        props.put("path", fName);
        output. setProperties(props);
        getProxyMXBean().createOutput(output);
        // Remove the tst_mb_aa_05_rate_CInt1 indicator to the ptst_mb_aa_05_BUG133_several_use_indicator probe definition
        JasmineProbe jp = getProxyMXBean().getProbe("ptst_mb_aa_05_BUG133_several_use_indicator");
        List<String> il = jp.getIndicatorList();
        il.remove("tst_mb_aa_05_rate_CInt1");
        jp.setIndicatorList(il);
        List<String> ol = new ArrayList<String>();
        ol.add(fName);
        ol.add("stdio"); // Help to find why this test does not work.
        jp.setOutputList(ol);
        jp.setPeriod(2);
        getProxyMXBean().changeProbe(jp);
        // Wait some time to do some collects and than stop the probe
        Thread.sleep(1000*10);
        getProxyMXBean().stopProbe("ptst_mb_aa_05_BUG133_several_use_indicator");
        // Check the collects
        CSVReader reader = null;
        int nbInt1 = 0;
        int nbInt2 = 0;
        int nbAdd = 0;
        int nbSub = 0;
        int nbDelta = 0;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_05_CInt1".equals(indicatorName)) {
                    nbInt1++;
                    Assert.assertEquals(Integer.parseInt(value), 15, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_05_CInt2".equals(indicatorName)) {
                    nbInt2++;
                    Assert.assertEquals(Integer.parseInt(value), 25, "Invalid  value for 'tst_mb_aa_05_CInt2' !");
                } else if ("tst_mb_aa_05_add_CInt1_1".equals(indicatorName)) {nbAdd++;
                    Assert.assertEquals(Integer.parseInt(value), 16, "Invalid  value for 'tst_mb_aa_05_add_CInt1_1' !");
                } else if ("tst_mb_aa_05_sub_CInt1_1".equals(indicatorName)) {
                    nbSub++;
                    Assert.assertEquals(Integer.parseInt(value), 14, "Invalid  value for 'tst_mb_aa_05_sub_CInt1_1' !");
                } else if ("tst_mb_aa_05_delta_CInt1".equals(indicatorName)) {
                    nbDelta++;
                    if (nbDelta > 1) {
                     // Do not check the first value because may be randomly different
                        Assert.assertEquals(Integer.parseInt(value), 0, "Invalid  value for 'tst_mb_aa_05_delta_CInt1' !");
                    }
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(nbInt2==0, "There is no collects for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbInt2, nbInt1, "There is not enough collected values for indicator  tst_mb_aa_05_CInt1 for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbAdd, nbInt1, "There is not enough collected values for indicator  tst_mb_aa_05_add_CInt1_1 for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbSub, nbInt1, "There is not enough collected values for indicator  tst_mb_aa_05_sub_CInt1_1 for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
        // TODO This test does not pass randomly with the error:
        //      'There is not enough collected values for indicator  tst_mb_aa_05_delta_CInt1'
        // So, just test there is some collected values for this indicator
        // Assert.assertEquals(nbDelta, nbInt1-1, "There is not enough collected values for indicator  tst_mb_aa_05_delta_CInt1 for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
        Assert.assertTrue(nbDelta > 0, "There is no collected values for indicator tst_mb_aa_05_delta_CInt1 for the probe 'ptst_mb_aa_05_BUG133_several_use_indicator' in the output " + fName);
    }

    /**
     * Test to reproduce the bug PROBE-145.
     * Check the collects for the 'mb_aa_05' mbean corresponding to the ptst_mb_aa_05_BUG145_several_use_indicator probe.
     */
    @Test
    public void testMb_aa_05_Bug145() throws Exception {
        logger.debug(">> testMb_aa_05_Bug145");
        String fName = getTmpDirPath() + "tst_mb_aa_05_BUG145_several_use_indicator.csv";
        CSVReader reader = null;
        boolean emptyOutputAdd = true;
        boolean emptyOutputSub = true;
        boolean emptyOutputTotal = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_05_CInt1".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 15, "Invalid 'CInt1' value !");
                } else if ("tst_mb_aa_05_add_CInt1_1".equals(indicatorName)) {
                    emptyOutputAdd = false;
                    Assert.assertEquals(Integer.parseInt(value), 16, "Invalid  value for 'tst_mb_aa_05_add_CInt1_1' !");
                } else if ("tst_mb_aa_05_sub_CInt1_1".equals(indicatorName)) {
                    emptyOutputSub = false;
                    Assert.assertEquals(Integer.parseInt(value), 14, "Invalid  value for 'tst_mb_aa_05_sub_CInt1_1' !");
                } else if ("tst_mb_aa_05_total".equals(indicatorName)) {
                    emptyOutputTotal = false;
                    Assert.assertEquals(Integer.parseInt(value), 30, "Invalid  value for 'tst_mb_aa_05_total' !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutputAdd, "There is no collects for the indicator 'tst_mb_aa_05_add_CInt1_1' in the output " + fName);
        Assert.assertFalse(emptyOutputSub, "There is no collects for the indicator 'tst_mb_aa_05_sub_CInt1_1' in the output " + fName);
        Assert.assertFalse(emptyOutputTotal, "There is no collects for the indicator 'tst_mb_aa_05_total' in the output " + fName);
    }

    /**
     * Check the collects for the 'mb_aa_06' mbean corresponding to the ptst_mb_aa_06_cache_several_use_indicator probe.
     * Check a collect is done only one time when a same indicator is used severals time in a same probe
     */
    @Test
    public void testMb_aa_06_cache_several_use_indicator() throws Exception {
        logger.debug(">> testMb_aa_06_cache_several_use_indicator");
        String fName = getTmpDirPath() + "tst_mb_aa_06_cache_several_use_indicator.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                if ("tst_mb_aa_06_CIncrement".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 20, "Invalid 'tst_mb_aa_06_CIncrement' value !");
                } else if ("tst_mb_aa_06_VIncremented".equals(indicatorName)) {
                } else if ("tst_mb_aa_06_delta_VIncremented".equals(indicatorName)) {
                    Assert.assertEquals(Integer.parseInt(value), 20, "Invalid  value for 'tst_mb_aa_06_delta_VIncremented' !");
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check DELTA, RATE, and SLOPE values on many mbeans.
     * Check the collects for the 'mb_bb_XX' mbeans corresponding to the ptst_mb_bb_xx_delta_rate_slope probe
     */
    @Test
    public void testMb_bb_XX_AttrDeltaRateSlope() throws Exception {
        boolean bFirstDeltaVI = true;
        boolean bFirstDeltaVS = true;
        boolean bFirstRateVI = true;
        boolean bFirstRateVS = true;
        boolean bFirstSlope1 = true;
        boolean bFirstSlope2 = true;

        logger.debug(">> testMb_bb_XX_AttrDeltaRateSlope");
        String fName = getTmpDirPath() + "tst_mb_bb_xx_delta_rate_slope.csv";
        CSVReader reader = null;
        boolean emptyOutput = true;
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            String value;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                emptyOutput = false;
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                value = nextLine[Util.CSV_INDEX_VALUE];
                // Check only the collect for the mbean named mb_bb_01
                if (nextLine[Util.CSV_INDEX_NAME].contains("type=Basic,category=bb,name=mb_bb_01")) {
                    if ("tst_mb_bb_xx_VPeriod".equals(indicatorName)) {
                        // Nothing to do. The VPeriod is not exactly the same value of the probe period
                    } else if ("tst_mb_bb_xx_CIncrement".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), 1, "Invalid 'CIncremented' value !");
                    } else if ("tst_mb_bb_xx_CAmplitude".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), 50, "Invalid 'CAmplitude' value !");
                    } else if ("tst_mb_bb_xx_CStep".equals(indicatorName)) {
                        Assert.assertEquals(Integer.parseInt(value), 2, "Invalid 'CStep' value !");
                    } else if ("tst_mb_bb_xx_delta_VIncremented".equals(indicatorName)) {
                        if ((bFirstDeltaVI) && (value.length() == 0)) {
                            // The first collect is the empty string
                        } else {
                            Assert.assertEquals(Double.parseDouble(value), 1.0, PRECISION_DOUBLE, "Invalid 'delta_VIncremented' value !");
                        }
                        bFirstDeltaVI = false;
                    } else if ("tst_mb_bb_xx_rate_VIncremented".equals(indicatorName)) {
                        if ((bFirstRateVI) && (value == null)) {
                            // The first collect is null or empty string
                        } else if ((bFirstRateVI) && (value.length() == 0)) {
                            // The first collect is null or empty string
                        } else {
                            Assert.assertTrue(between(value, (1.0/(THEO_PERIOD + VPERIOD_ACCEPTED_DELTA)), (1.0/(THEO_PERIOD - VPERIOD_ACCEPTED_DELTA))), "Invalid 'rate_VIncremented' value " + value);
                        }
                        bFirstRateVI = false;
                    } else if ("tst_mb_bb_xx_delta_VSinusoidal".equals(indicatorName)) {
                        if ((bFirstDeltaVS) && (value == null)) {
                            // The first collect is null or empty string
                        } else if ((bFirstDeltaVS) && (value.length() == 0)) {
                            // The first collect is null or empty string
                        } else {
                            Assert.assertTrue(between(value, -2, 2), "Invalid 'delta_VSinusoidal' value ! (-2,+2," + value + ")");
                        }
                        bFirstDeltaVS = false;
                    } else if ("tst_mb_bb_xx_rate_VSinusoidal".equals(indicatorName)) {
                        if ((bFirstRateVS) && (value == null)) {
                            // The first collect is null or empty string
                        } else if ((bFirstRateVS) && (value.length() == 0)) {
                            // The first collect is null or empty string
                        } else {
                            Assert.assertTrue(between(value, (2.0/(THEO_PERIOD + VPERIOD_ACCEPTED_DELTA)), (2.0/(THEO_PERIOD - VPERIOD_ACCEPTED_DELTA))), "Invalid 'rate_VSinusoidal' value " + value);
                        }
                        bFirstRateVS = false;
                    } else if ("tst_mb_bb_xx_VIncremented_per_VSinusoidal".equals(indicatorName)) {
                        if ((bFirstSlope1) && (value == null)) {
                            // The first collect is null or empty string
                        } else if ((bFirstSlope1) && (value.length() == 0)) {
                            // The first collect is null or empty string
                        } else {
                            Double dAbsValue = StrictMath.abs(Double.parseDouble(value));
                            Assert.assertEquals(dAbsValue, 0.5, PRECISION_DOUBLE, "Invalid 'VIncremented_per_VSinusoidal' value !");
                        }
                        bFirstSlope1 = false;
                    } else if ("tst_mb_bb_xx_CAmplitude_per_CStep".equals(indicatorName)) {
                        if ((bFirstSlope2) && (value == null)) {
                            // The first collect is null or empty string
                        } else if ((bFirstSlope2) && (value.length() == 0)) {
                            // The first collect is null empty string
                        } else {
                            // TODO: The previous value with the version 1.3 was
                            // 0.0
                            // Assert.assertEquals(Double.parseDouble(value),
                            // 0.0,
                            // PRECISION_DOUBLE,
                            // "Invalid 'CAmplitude_per_CStep' value !");
                            Assert.assertEquals(Double.parseDouble(value), 1.0, PRECISION_DOUBLE, "Invalid 'CAmplitude_per_CStep' value !");
                        }
                        bFirstSlope2 = false;
                    } else if ("tst_mb_bb_xx_VIncremented".equals(indicatorName)
                            || "tst_mb_bb_xx_VSinusoidal".equals(indicatorName)) {
                        // do nothing
                    } else {
                        Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                    }
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(emptyOutput, "There is no collects in the output " + fName);
    }

    /**
     * Check the collects corresponding to the ptst_df_all_several_use_indicator probe.
     */
    @Test
    public void testDfAllSeveralUseIndicator() throws Exception {
        logger.debug(">> testDfAllSeveralUseIndicator");
        String fName = getTmpDirPath() + "tst_df_all_several_use_indicator.csv";
        CSVReader reader = null;
        int nbDfAll = 0;
        int nbDfAllMin = 0;
        int nbDfAllMax = 0;
        int nbDfAllAddMinMax = 0;
        int nbDfAllMulMinMax = 0;
        List<String> diskNames = new ArrayList<String>();
        try {
            reader = new CSVReader(new FileReader(fName), ';');
            String[] nextLine;
            String indicatorName;
            while ((nextLine = reader.readNext()) != null) {
                // nextLine[] is an array of values from the line
                indicatorName =  Util.getNameOfFullName(nextLine[Util.CSV_INDEX_NAME]);
                if (indicatorName.startsWith("tst_df_all.")) {
                    nbDfAll++;
                    String diskName = indicatorName.substring("tst_df_all.".length()-1);
                    if (!diskNames.contains(diskName)) {
                        diskNames.add(diskName);
                    }
                } else if ("tst_df_all_min".equals(indicatorName)) {
                    nbDfAllMin++;
                } else if ("tst_df_all_max".equals(indicatorName)) {
                    nbDfAllMax++;
                } else if ("tst_df_all_add_min_max".equals(indicatorName)) {
                    nbDfAllAddMinMax++;
                } else if ("tst_df_all_mul_min_max".equals(indicatorName)) {
                    nbDfAllMulMinMax++;
                } else {
                    Assert.fail("The '" + indicatorName + "' indicator shoud not be collected !");
                }
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
        Assert.assertFalse(nbDfAll==0, "There is no collects for the indicator 'tst_df_all' in the output " + fName);
        int nbCollects = nbDfAll / diskNames.size();
        Assert.assertEquals(nbCollects, nbDfAllMin, "There is not enough collected values for indicator 'tst_df_all_min' for the probe 'ptst_df_all_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbCollects, nbDfAllMax, "There is not enough collected values for indicator 'tst_df_all_max' for the probe 'ptst_df_all_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbCollects, nbDfAllAddMinMax, "There is not enough collected values for indicator 'tst_df_all_add_min_max' for the probe 'ptst_df_all_several_use_indicator' in the output " + fName);
        Assert.assertEquals(nbCollects, nbDfAllMulMinMax, "There is not enough collected values for indicator 'tst_df_all_mul_min_max' for the probe 'ptst_df_all_several_use_indicator' in the output " + fName);
    }


    private static boolean between(String x, int a, int b) {
        int xi = Integer.parseInt(x);
        // logger.debug(">>>> x = " + xi + ", a = " + a + ", b = " + b);
        return ((a <= xi) && (xi <= b));
    }

    private static boolean between(String x, Double a, Double b) {
        double dx = Double.parseDouble(x);
        // logger.debug(">>>> x = " + dx + ", a = " + a + ", b = " + b);
        return ((a - PRECISION_DOUBLE < dx) && (dx < b + PRECISION_DOUBLE));
    }


}
