/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.linux.internal;

import org.ow2.jasmine.probe.collectors.linux.LinuxCollectingException;
import org.ow2.jasmine.probe.collectors.linux.NoProcResourceException;

/**
 * Collector allowing to get measurements about cpu activity.
 * Based on Lewys projetc's CpuProbe.
 * Uses data from /proc/stat file
 *
 * @author danesa
 *
 */
public class CpuCollector extends ResourceCollector {

    private final static String CPU_USER =   "user";
    private final static String CPU_NICE=    "nice";
    private final static String CPU_KERNEL = "kernel";
    private final static String CPU_IDLE=    "idle";
    // resource name must start with CPU, and then cpu number
    private final static String CPU = "cpu";

    // nb of supported parameters
    private final static int NB_OF_PARAMS  = 4;

    // the 'i' in 'cpui'
    private int cpuNb;

    public CpuCollector(final String resourceName, final String fileName) throws NoProcResourceException {
        super(resourceName, fileName, NB_OF_PARAMS);
        int index = resourceName.indexOf(CPU);
        if (index < 0 || index > 0)
            throw new NoProcResourceException("Invalid resource name: " + resourceName + "(should start with " + CPU + ")");
        String cpuNbString = resourceName.substring(CPU.length());
        if (cpuNbString.length() == 0) {
            // The asked resource is 'cpu' (which aggregates all the cpuNb lines)
            cpuNb = -1;
        } else {
            cpuNb = Integer.parseInt(cpuNbString);
        }
        params[0] = CPU_USER;
        params[1] = CPU_NICE;
        params[2] = CPU_KERNEL;
        params[3] = CPU_IDLE;
        checkFormat(buffer);
    }

    /**
     * Verify the expected format:
     * <code>
     * cpu userTime niceTime sysTime idleTime
     * cpu0 userTime niceTime sysTime idleTime
     * cpu1 userTime niceTime sysTime idleTime
     * ...
     * </code>
     * @param buffer
     * @throws NoProcResourceException
     */
    private void checkFormat(final byte[] buffer) throws NoProcResourceException {
        int i = 0;
        do {
            if (buffer[i++] != 'c')
                break;
            if (buffer[i++] != 'p')
                throw invalidFileFormat;
            if (buffer[i++] != 'u')
                throw invalidFileFormat;
            // Skip cpu number
            int _cpuNb = 0;
            while (buffer[i] != ' ' && buffer[i] >= '0' && buffer[i] <= '9')
            {
                _cpuNb = _cpuNb * 10 + buffer[i] - '0';
                i++;
            }
            // Skip spaces
            while (buffer[i] == ' ')
                i++;
            // Check for user/nice/sys/idle values
            for (int cpuStat = 0; cpuStat < NB_OF_PARAMS; cpuStat++) {
                // Skip values
                while ((buffer[i] >= '0') && (buffer[i] <= '9'))
                    i++;
                if ((buffer[i] != ' ') && (buffer[i] != '\n') && (buffer[i] != '\r'))
                    throw invalidFileFormat;
                while (buffer[i] == ' ')
                    i++;
            }
            // Skip to the next line
            while (buffer[i] != '\n')
                i++;
            i++; // Consume '\n'
        } while (i < BUFFER_SIZE);
    }

    /**
     * Read parameters' values from the file/buffer and set values[].
     * @throws LinuxCollectingException
     */
    protected void getUsage() throws LinuxCollectingException {
        try {
            procFile.seek(0);
            procFile.read(buffer);
            int i = 0;
            int nb = 0; // current cpu line number
            long foo;
            int index = 0;// values index
            do {
                if (buffer[i++] != 'c')
                    break;
                if (buffer[i++] != 'p')
                    throw invalidFileFormat;
                if (buffer[i++] != 'u')
                    throw invalidFileFormat;
                // current cpu line is line we have to read values from
                if (nb == (cpuNb + 1)) {
                    // read values and break out
                    if (buffer[i++] != ' ') {
                        // Skip cpu number
                        while (buffer[i] >= '0' && buffer[i] <= '9')
                            i++;
                    }
                    // Skip spaces
                    while (buffer[i] == ' ')
                        i++;
                    // Check for user/nice/system/idle values
                    for (int cpuStat = 0; cpuStat < NB_OF_PARAMS; cpuStat++) {
                        // read value
                        foo = 0;
                        while ((buffer[i] >= '0') && (buffer[i] <= '9')) {
                            foo = foo * 10 + buffer[i] - '0';
                            i++;
                        }
                        values[index++] = foo;
                        // Skip spaces
                        while (buffer[i] == ' ')
                            i++;
                    }
                    break;
                } else {
                    nb++;
                    // skip this line and continue
                    while (buffer[i] != '\n')
                        i++;
                    i++; // Consume '\n'
                }
            } while (i < BUFFER_SIZE);
        } catch (Exception e) {
            throw new LinuxCollectingException(e);
        }
    }
}
