/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.Consumes;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.ow2.jasmine.probe.api.generated.IndicatorBaseType;

/**
 * Root resource class for indicators management.
 * Contains methods to treat HTTP requests on "indicator" resources.
 * @author danesa
 *
 */
@Path("/indicator")
public interface Indicators {

    public final String RESOURCE_NAME = "indicator";

    /**
     * Process HTTP GET requests.
     * Possible URIs are:
     *  /indicator that corresponds to 'indicator-list' command
     * or
     * /indicator?type={typeName} that corresponds to 'indicator-list -t typeName' command
     * or
     * /indicator?verbose="true" that corresponds to 'indicator-list -v' command
     * @return response containing the existent indicators
     */
    @GET
    @Produces(MediaType.APPLICATION_XML)
    Response getIndicators(@QueryParam("type") String type
            , @DefaultValue("false") @QueryParam("verbose") boolean verbose
            , @Context UriInfo ui);


    /**
     * Process HTTP GET requests having /indicator/{name} URI
     * Correspond to indicator-list -n name
     * @param name indicator name
     * @param type
     * @return response
     */
    @GET
    @Path("/{name}")
    @Produces(MediaType.APPLICATION_XML)
    Response getIndicator(@PathParam("name") String name, @Context UriInfo ui);

    /**
     * Process HTTP POST requests having /indicator URI and new indicator definition in its body.
     * Corresponds to indicator-create command
     * @param indicatorDef new indicator definition
     * @return
     */
    @POST
    @Consumes(MediaType.APPLICATION_XML)
    @Produces(MediaType.APPLICATION_XML)
    Response createIndicator(IndicatorBaseType indicatorDef);


    /**
     * Process HTTP PUT requests having /indicator/{name} URI and new indicator definition in its body.
     * Corresponds to indicator-change -n name command.
     *
     * @param indicatorDef updated indicator definition
     * @param name indicator name
     * @return response with state OK or NOT_FOUND or BAD_REQUEST if change failed
     */
    @PUT
    @Path("/{name}")
    @Consumes(MediaType.APPLICATION_XML)
    @Produces(MediaType.APPLICATION_XML)
    Response changeIndicator(IndicatorBaseType indicatorDef, @PathParam("name") String name);


    /**
     * Process HTTP DELETE requests having /indicator/{name} URI
     * Corresponds to indicator-remove name
     * @param name the name of the indicator to remove
     */
    @DELETE
    @Path("/{name}")
    @Produces(MediaType.APPLICATION_XML)
    Response deleteIndicator(@PathParam("name") String name);
}
