/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.probescheduler.internal;

import java.util.ArrayList;
import java.util.List;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.probescheduler.*;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

@Component(name="JasmineInternalScheduler")
@Provides
public class SchedulerServiceImpl implements SchedulerService {

    private Log logger = LogFactory.getLog(SchedulerService.class);

    // threads managing the service.
    private static Batch batchThread;
    private static Clock clockThread;

    private final static long PERIOD_MAX = 30000; // 30 sec
    private final static long PERIOD_MIN = 100; // 1/10 sec
    private long period;
    private long minremtime = PERIOD_MAX;

    private ArrayList<TaskReferenceImpl> timerList = new ArrayList<TaskReferenceImpl>();
    private ArrayList<TaskReferenceImpl> expiredList = new ArrayList<TaskReferenceImpl>();

    private static boolean shuttingdown = false;

    /**
     * Start the ipojo
     */
    @Validate
    public void start() {
        logger.debug("");
        // launch threads for timers
        batchThread = new Batch(this);
        batchThread.start();
        clockThread = new Clock(this);
        clockThread.start();
    }

    /**
     * Stop the ipojo
     */
    @Invalidate
    public void stop() {
        logger.debug("");
    }

    public TaskReference schedulePeriodicTask(String name, Runnable task,
            long initialDelay, long period) throws SchedulerException {
        logger.debug("");
        TaskReferenceImpl timer = new TaskReferenceImpl(task, period, true);
        synchronized(timerList) {
            timerList.add(timer);
            if (period < minremtime) {
                minremtime = period;
            }
            timerList.notifyAll();
        }
        return timer;
    }

    /**
     * update all timers in the list
     * each timer expired is put in a special list of expired timers
     * they will be processed then by the Batch Thread.
     */
    public void clock() {
        synchronized(timerList) {
            while (true) {
                // compute time to sleep
                if (shuttingdown) {
                    period = 1;
                } else {
                    period = PERIOD_MAX;
                    if (minremtime < period) {
                        period = minremtime < PERIOD_MIN ? PERIOD_MIN : minremtime;
                    }
                }
                // must sleep a little less than period
                try {
                    timerList.wait(period);
                } catch (InterruptedException e) {
                    logger.error("Timer interrupted");
                }
                int found = 0;
                boolean empty = true;
                minremtime = PERIOD_MAX;
                List<TaskReferenceImpl> toremove = new ArrayList<TaskReferenceImpl>();
                for (TaskReferenceImpl t : timerList) {
                    if (!t.isStopped()) {
                        empty = false;
                    }
                    long remtime = t.update();
                    if (remtime <= 0) {
                        if (t.valid()) {
                            expiredList.add(t);
                            found++;
                            if (t.ispermanent() && !shuttingdown) {
                                remtime = t.restart();
                                if (remtime < minremtime) {
                                    minremtime = remtime;
                                }
                            } else {
                                toremove.add(t);
                            }
                        } else {
                            toremove.add(t);
                        }
                    } else {
                        if (remtime < minremtime) {
                            minremtime = remtime;
                        }
                    }
                    // Be sure there is no more ref on bean in this local variable.
                    t = null;
                }
                for (TaskReferenceImpl t : toremove) {
                    timerList.remove(t);
                }
                if (found > 0) {
                    timerList.notifyAll();
                } else {
                    if (empty) {
                        if (shuttingdown) {
                            break;
                        }
                    }
                }
            }
            timerList.notifyAll();
        }
    }

    /**
     * process all expired timers
     */
    public void batch() {

        while (!(shuttingdown && timerList.isEmpty() && expiredList.isEmpty())) {
            TaskReferenceImpl t;
            synchronized(timerList) {
                while (expiredList.isEmpty()) {
                    if (shuttingdown) {
                        logger.warn("TimerManager shutting down");
                        return;
                    }
                    try {
                        timerList.wait();
                    } catch (Exception e) {
                        logger.error("Exception in Batch: ",e);
                    }
                }
                t = (TaskReferenceImpl) expiredList.remove(0);
            }
            // Do not keep the lock during the processing of the timer
            try {
                t.process();
            } catch (NullPointerException e) {
                e.printStackTrace();
                logger.warn("Ignoring null pointer exception");
            } catch (Exception e) {
                // An exception in a timer process should not stop this thread.
                e.printStackTrace();
                logger.warn("Ignoring exception: " + e);
            }
        }
        logger.info("TimerManager stopped");
    }

}

/**
 * Clock thread for a TimerManager
 * Every second, decrement timers and launch action if expired
 */
class Clock extends Thread {

    private SchedulerServiceImpl tmgr;

    public Clock(SchedulerServiceImpl tmgr) {
        super("JasmineClock");
        this.tmgr = tmgr;
    }

    public void run() {
        tmgr.clock();
    }
}

/**
 * Batch thread for a TimerManager
 * pocess all expired timers
 */
class Batch extends Thread {

    private SchedulerServiceImpl tmgr;

    public Batch(SchedulerServiceImpl tmgr) {
        super("JasmineScheduler");
        this.tmgr = tmgr;
    }

    public void run() {
        tmgr.batch();
    }
}
