/**
 * JASMINe
 * Copyright (C) 2013 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

package org.ow2.jasmine.probe.shell;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.service.command.CommandSession;
import org.ow2.jasmine.probe.JasmineObject;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.Collection;

/**
 * Shell command: target-list
 *
 * @author durieuxp
 */
@Component
@Command(name = "target-list",
        scope = "jasmine",
        description = "List JASMINe targets")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie'/>")
public class TargetList implements Action {

    @Requires
    JasmineProbeManager probeManager = null;

    @Argument(index = 0, name = "name", description = "Target Name", required = false, multiValued = false)
    private String name;

    @Option(name = "-t", aliases = "--type", description = "Target Type", required = false, multiValued = false)
    private String type = null;

    @Option(name = "-v", aliases = "--verbose", description = "If true, print details about targets", required = false, multiValued = false)
    private Boolean verbose = false;

    protected static Log logger = LogFactory.getLog(Util.class);

    public Object execute(CommandSession commandSession) {

        logger.debug("");
        try {
            if (name == null) {
                // List all targets
                Collection<JasmineTarget> objs = probeManager.getTargets();
                if (objs.isEmpty()) {
                    String message1 = "There are no JasmineTargets";
                    String message2 = (type != null ? " of type " + type : "");
                    logger.error(message1 + message2);
                    return null;
                }
                for (JasmineObject obj : objs) {
                    if (verbose) {
                        System.out.println(obj.toLine());
                    } else {
                        System.out.println(obj.getName());
                    }
                }
            } else {
                // Show one Object
                JasmineObject obj = probeManager.getTarget(name);
                System.out.println(obj.toString());
            }
        } catch (Exception e) {
            logger.error("target-list failed: ", e);
        }
        return null;
    }

}