/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.itests.rest;

import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.WebResource;

import java.io.File;
import java.net.HttpURLConnection;
import java.util.List;

import org.ow2.jasmine.probe.api.generated.IndicatorResource;
import org.ow2.jasmine.probe.api.generated.OutputResource;
import org.ow2.jasmine.probe.api.generated.ProbeResource;
import org.ow2.jasmine.probe.api.generated.PropInfoType;
import org.ow2.jasmine.probe.api.generated.PropType;
import org.ow2.jasmine.probe.api.generated.ResourceType;
import org.ow2.jasmine.probe.api.generated.Status;
import org.ow2.jasmine.probe.api.generated.TargetResource;

import org.testng.Assert;
import org.testng.annotations.Test;

public class GetTest extends AbsRestTest {

    // See probe-config.xml to get the correct value.
    final public static int PROBE_CPU_PERIOD = 9;


    /**
     * Get probes list
     */
    @Test
    public void getProbes() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get probes list
        String sUri = urlBase + "/probe";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the probes list: 3 probes named 'poll-cpusun' and 'pThreading', and 'pData'
        List<ProbeResource> probes = response.getEntity(new GenericType<List<ProbeResource>>() {});
        Assert.assertEquals(probes.size(), 4, "Incorrect probes number");
        for (ProbeResource probe : probes) {
            if (!"poll-cpusun".equals(probe.getId())
                    && !"pThreading".equals(probe.getId())
                    && !"pData".equals(probe.getId())
                    && !"pChange".equals(probe.getId())) {
                Assert.fail("Incorrect probe id (" + probe.getId() + ")");
            } else {
                Assert.assertEquals(probe.getHref(), sUri+"/"+probe.getId(), "Incorrect probe href");
            }
        }
    }

    /**
     * Get 'poll-cpusun' probe
     */
    @Test
    public void getProbeCpu() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'poll-cpusun' probe
        String sUri = urlBase + "/probe/poll-cpusun";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check
        ProbeResource probe = response.getEntity(ProbeResource.class);
        Assert.assertEquals(probe.getId(), "poll-cpusun", "Incorrect probe id");
        Assert.assertEquals(probe.getHref(), sUri, "Incorrect probe href");
        Assert.assertEquals(probe.getPeriod().intValue(), PROBE_CPU_PERIOD, "Incorrect period");
        List<String> indicators = probe.getIndicator();
        Assert.assertEquals(indicators.size(), 3, "Incorrect indicators list (" + indicators.toString() + ")");
        Assert.assertTrue(indicators.contains("processCpuTime"), "Indicators list not contains 'processCpuTime' (" + indicators.toString() + ")");
        Assert.assertTrue(indicators.contains("currentCpuTime"), "Indicators list not contains 'currentCpuTime' (" + indicators.toString() + ")");
        Assert.assertTrue(indicators.contains("currentCpuLoad"), "Indicators list not contains 'currentCpuLoad' (" + indicators.toString() + ")");
        List<String> outputs = probe.getOutput();
        Assert.assertEquals(outputs.size(), 1, "Incorrect outputs list (" + outputs.toString() + ")");
        Assert.assertTrue(outputs.contains("otst_poll_cpusun"), "Outputs list not contains 'otst_poll_cpusun' (" + outputs.toString() + ")");
    }

    /**
     * Get 'poll-cpusun' probe and check collects
     */
    @Test
    public void getProbeCpuCheckCollects() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'poll-cpusun' probe
        String sUri = urlBase + "/probe/poll-cpusun";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug mode
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check
        ProbeResource probe = response.getEntity(ProbeResource.class);
        Assert.assertEquals(probe.getId(), "poll-cpusun", "Incorrect probe id");
        Assert.assertEquals(probe.getStatus(), Status.RUNNING, "Incorrect probe status");
        String fNameOutput = tmpDirPath + "otst_poll_cpusun.csv";
        File fOutput = new File(fNameOutput);
        Assert.assertFalse(fOutput.length() == 0, "Output file for 'poll-cpusun' probe not exist or empty ("+fNameOutput+")");
    }

    /**
     * Get 'unknown' probe
     */
    @Test
    public void getProbeUnknown() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'unknown' probe
        String sUri = urlBase + "/probe/Unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        }
    }

    /**
     * Get indicators list
     */
    @Test
    public void getIndicators() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get indicators list
        String sUri = urlBase + "/indicator";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the indicators list: 10 indicators named .....
        List<IndicatorResource> indicators = response.getEntity(new GenericType<List<IndicatorResource>>() {});
        Assert.assertEquals(indicators.size(), 11, "Incorrect indicators number");
        for (IndicatorResource indicator : indicators) {
            if (!"C_1000_000".equals(indicator.getName())
                    && !"C_data_5".equals(indicator.getName())
                    && !"C_data_10".equals(indicator.getName())
                    && !"_procnb".equals(indicator.getName())
                    && !"_cputime".equals(indicator.getName())
                    && !"currentCpuTime".equals(indicator.getName())
                    && !"processCpuTime".equals(indicator.getName())
                    && !"_rate".equals(indicator.getName())
                    && !"currentCpuLoad".equals(indicator.getName())
                    && !"threading".equals(indicator.getName())
                    && !"C_change_1".equals(indicator.getName())) {
                Assert.fail("Incorrect indicator name (" + indicator.getName() + ")");
            } else {
                Assert.assertEquals(indicator.getHref(), sUri+"/"+indicator.getName(), "Incorrect indicator href");
            }
        }
    }

    /**
     * Get correlate indicators list
     */
    @Test
    public void getIndicatorsCorrelate() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'correlate' indicators list
        String sUri = urlBase + "/indicator?type=correlate";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the indicators list: 2 'correlate' indicators named 'processCpuTime' and 'currentCpuLoad'
        List<IndicatorResource> indicators = response.getEntity(new GenericType<List<IndicatorResource>>() {});
        Assert.assertEquals(indicators.size(), 2, "Incorrect correlate indicators number");
        for (IndicatorResource indicator : indicators) {
            if (!"processCpuTime".equals(indicator.getName()) && !"currentCpuLoad".equals(indicator.getName())) {
                Assert.fail("Incorrect correlate indicators names ("+ indicator.getName() + ")");
            } else {
                Assert.assertEquals(indicator.getHref(), urlBase+"/indicator/"+indicator.getName(), "Incorrect indicator href");
            }
        }
    }

    /**
     * Get 'threading' indicator
     */
    @Test
    public void getIndicatorThreading() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'threading' indicator
        String sUri = urlBase + "/indicator/threading";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the indicator
        IndicatorResource indicator = response.getEntity(IndicatorResource.class);
        Assert.assertEquals(indicator.getName(), "threading", "Incorrect indicator name");
        Assert.assertEquals(indicator.getHref(), sUri, "Incorrect indicator href");
        Assert.assertEquals(indicator.getScale(), 1, "Incorrect indicator scale");
        Assert.assertEquals(indicator.getType(), "jmx", "Incorrect indicator type");
        List<PropType> iProps = indicator.getProperty();
        Assert.assertEquals(iProps.size(), 3, "Incorrect properties indicator");
        for (PropType p : iProps) {
            if ("target".equals(p.getKey())) {
                Assert.assertEquals(p.getValue(), "jprobe", "Incorrect property target");
            } else if ("mbean".equals(p.getKey())) {
                Assert.assertEquals(p.getValue(), "java.lang:type=Threading", "Incorrect property mbean");
            } else if ("attr".equals(p.getKey())) {
                Assert.assertEquals(p.getValue(), "ThreadCount,TotalStartedThreadCount", "Incorrect property attr");
            } else {
                Assert.fail("Incorrect property key (" + p.getKey() + ")");
            }
        }
        List <String> iSources = indicator.getSource();
        Assert.assertTrue(iSources.isEmpty(), "Incorrect sources indicator: not empty");
    }

    /**
     * Get 'unknown' indicator
     */
    @Test
    public void getIndicatorUnknown() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'unknown' indicator
        String sUri = urlBase + "/indicator/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        }
    }

    /**
     * Get indicator types
     */
    @Test
    public void getIndicatorTypes() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get indicator types list
        String sUri = urlBase + "/indicatorType";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the indicator types: 8 indicator types .....
        List<ResourceType> iTypes = response.getEntity(new GenericType<List<ResourceType>>() {});
        Assert.assertEquals(iTypes.size(), 9, "Incorrect indicator types list");
        for (ResourceType iType : iTypes) {
            if (!"aggregate".equals(iType.getType())
                    && !"constant".equals(iType.getType())
                    && !"correlate".equals(iType.getType())
                    && !"derived".equals(iType.getType())
                    && !"df".equals(iType.getType())
                    && !"jmx".equals(iType.getType())
                    && !"lewys".equals(iType.getType())
                    && !"merge".equals(iType.getType())
                    && !"slope".equals(iType.getType())) {
                Assert.fail("Incorrect indicator type (" + iType.getType() + ")");
            }
        }
    }

    /**
     * Get 'constant' indicator properties
     */
    @Test
    public void getIndicatorConstantProps() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'constant' indicator properties
        String sUri = urlBase + "/indicatorType/constant/props";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the 'constant' indicator properties.....
        List<PropInfoType> iProps = response.getEntity(new GenericType<List<PropInfoType>>() {});
        Assert.assertEquals(iProps.size(), 2, "Incorrect 'constant' indicator properties list");
        for (PropInfoType iProp : iProps) {
            if (!"type".equals(iProp.getName())
                    && !"value".equals(iProp.getName())) {
                Assert.fail("Incorrect 'constant' indicator property name (" + iProp.getName() + ")");
            }
            Assert.assertEquals(iProp.getRequired(), "true", "Incorrect indicator properties (not required)");
        }
    }

    /**
     * Get 'jmx' indicator info
     */
    @Test
    public void getIndicatorJmxInfo() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'jmx' indicator info
        String sUri = urlBase + "/indicatorType/jmx/info";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
    }

    /**
     * Get outputs list
     */
    @Test
    public void getOutputs() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get output list
        String sUri = urlBase + "/output";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the output list: 4 indicators named 'stdio1', 'otst_poll_cpusun','otst_threading' and 'otst_data'
        List<OutputResource> outputs = response.getEntity(new GenericType<List<OutputResource>>() {});
        Assert.assertEquals(outputs.size(), 5, "Incorrect outputs number");
        for (OutputResource output : outputs) {
            if (!"stdio1".equals(output.getName())
                    && !"otst_poll_cpusun".equals(output.getName())
                    && !"otst_threading".equals(output.getName())
                    && !"otst_data".equals(output.getName())
                    && !"otst_change".equals(output.getName())) {
                Assert.fail("Incorrect output name (" + output.getName() + ")");
            } else {
                Assert.assertEquals(output.getHref(), sUri+"/"+output.getName(), "Incorrect output href");
            }
        }
    }

    /**
     * Get 'otst_threading' output
     */
    @Test
    public void getOutputThreading() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'otst_threading' output
        String sUri = urlBase + "/output/otst_threading";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the output
        OutputResource output = response.getEntity(OutputResource.class);
        Assert.assertEquals(output.getName(), "otst_threading", "Incorrect output name");
        Assert.assertEquals(output.getHref(), sUri, "Incorrect output href");
        Assert.assertEquals(output.getType(), "file", "Incorrect output type");
    }

    /**
     * Get 'unknown' output
     */
    @Test
    public void getOutputUnknown() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'unknown' output
        String sUri = urlBase + "/output/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        }
    }

    /**
     * Get output types
     */
    @Test
    public void getOutputTypes() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get output types list
        String sUri = urlBase + "/outputType";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the output types: 4 output types .....
        List<ResourceType> oTypes = response.getEntity(new GenericType<List<ResourceType>>() {});
        Assert.assertEquals(oTypes.size(), 4, "Incorrect output types list");
        for (ResourceType oType : oTypes) {
            if (!"console".equals(oType.getType())
                    && !"file".equals(oType.getType())
                    && !"mule".equals(oType.getType())
                    && !"rest".equals(oType.getType())) {
                Assert.fail("Incorrect output type (" + oType.getType() + ")");
            }
        }
    }

    /**
     * Get 'mule' output properties
     */
    @Test
    public void getOuputMuleProps() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'mule' ouput properties
        String sUri = urlBase + "/outputType/mule/props";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the 'mule' output properties.....
        List<PropInfoType> oProps = response.getEntity(new GenericType<List<PropInfoType>>() {});
        Assert.assertEquals(oProps.size(), 1, "Incorrect 'mule' ouput properties list");
        for (PropInfoType oProp : oProps) {
            if (!"url".equals(oProp.getName())) {
                Assert.fail("Incorrect 'mule' indicator property name (" + oProp.getName() + ")");
            }
            Assert.assertEquals(oProp.getRequired(), "false", "Incorrect output properties (required)");
        }
    }

    /**
     * Get targets list
     */
    @Test
    public void getTargets() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get targets list
        String sUri = urlBase + "/target";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the target list: 1 target named 'jprobe'
        List<TargetResource> targets = response.getEntity(new GenericType<List<TargetResource>>() {});
        Assert.assertEquals(targets.size(), 1, "Incorrect target number");
        for (TargetResource target : targets) {
            if (!"jprobe".equals(target.getName())) {
                Assert.fail("Incorrect target name (" + target.getName() + ")");
            } else {
                Assert.assertEquals(target.getHref(), sUri+"/"+target.getName(), "Incorrect target href");
            }
        }
    }

    /**
     * Get 'jprobe' target
     */
    @Test
    public void getTargetJProbe() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'jprobe' target
        String sUri = urlBase + "/target/jprobe";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_OK, "Bad HTTP status");
        }
        // Check the target
        TargetResource target = response.getEntity(TargetResource.class);
        Assert.assertEquals(target.getName(), "jprobe", "Incorrect target name");
        Assert.assertEquals(target.getHref(), sUri, "Incorrect target href");
        List<PropType> tProps = target.getProperty();
        Assert.assertEquals(tProps.size(), 1, "Incorrect properties list");
        for (PropType p : tProps) {
            if (!"url".equals(p.getKey())) {
                Assert.fail("Incorrect property (" + p.getKey() + ")");
            }
        }
    }

    /**
     * Get 'unknown' target
     */
    @Test
    public void getTargetUnknown() throws Exception {
        WebResource webResource;
        ClientResponse response;

        // Get 'unknown' target
        String sUri = urlBase + "/target/unknown";
        logger.info("Resource URI: " + sUri);
        webResource = restClient.resource(sUri);
        response = webResource.get(ClientResponse.class);
        // Check the status of the request
        Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        // If debug, trace the response
        if (logger.isDebugEnabled()) {
            String output = response.getEntity(String.class);
            logger.debug("  Response: " + output);
            // Redo the request
            response = webResource.get(ClientResponse.class);
            Assert.assertEquals(response.getStatus(), HttpURLConnection.HTTP_NOT_FOUND, "Bad HTTP status");
        }
    }

}
