/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.aggregate.internal;

import java.util.Collection;
import java.util.HashMap;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineAggregateService;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;

/**
 * Aggregate collector implementation.
 * @author danesa
 */
public class JAggregateCollector extends JCollector {

    /**
     * Built-in aggregate operation code.
     */
    private int ope;

    /**
     * Requested operation (aggregate function).
     */
    private String operation;

    /**
     * List of supported operations
     */
    public final static int OP_MIN = 1;
    public final static int OP_MAX = 2;
    public final static int OP_SUM = 3;
    public final static int OP_AVERAGE = 4;

    private JasmineCollector source = null;

    /**
     * Name of the field to merge
     */
    private String merged;

    /**
     * Constructor
     * @param probeId
     * @param def
     * @param period
     * @param source
     */
    public JAggregateCollector(String probeId, JasmineIndicator def, int period, String operation, JasmineCollector source, String merged) {
        super(probeId, def, period);
        this.operation = operation;
        this.ope = 0;
        if (operation.equalsIgnoreCase("min")) {
            this.ope = OP_MIN;
        } else if (operation.equalsIgnoreCase("max")) {
            this.ope = OP_MAX;
        }  else if (operation.equalsIgnoreCase("sum")) {
            this.ope = OP_SUM;
        }  else if (operation.equalsIgnoreCase("average")) {
            this.ope = OP_AVERAGE;
        }
        this.source = source;
        this.merged = merged;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        JasmineIndicatorValue current = source.getLastResult();
        if (current == null) {
            logger.warn("No result available on source indicator {0}", source.getIndicatorName());
            return null;
        }

        JasmineAggregateService aggserv = probeManager.getAggregate(operation);
        if (aggserv != null) {
            // Got a service to do aggregation
            return aggserv.aggregate(current);
        }

        /*
         * Otherwise, do operation
         */
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());

        // Keep metadata from source indicator (server, domain, ...)
        for (String key : current.getMetadata().keySet()) {
             jiv.addMetadata(key, current.getMetadata().get(key));
        }
        jiv.setTarget(current.getTarget());

        Collection<JasmineSingleResult> jsrs = current.getValues();

        // Props for JSR to construct
        HashMap<String, String> jsrProps = jsrProps(jsrs);

        JasmineSingleResult jsrm = new JasmineSingleNumberResult();

        Number resval = null;
        switch (ope) {
            case OP_MAX:
                resval = maxJsr(jsrs);
                break;
            case OP_MIN:
                resval = minJsr(jsrs);
                break;
            case OP_SUM:
                resval = addJsr(jsrs);
                break;
            case OP_AVERAGE:
                resval = averageJsr(jsrs);
                break;
            default:
                throw new JasmineCollectorException("Unimplemented merge operation " + ope);
        }
        jsrm.setName("");
        jsrm.setTimestamp(current.getTimestamp());
        jsrm.setValue(resval);
        jsrm.setProperties(jsrProps);
        jiv.addValue(jsrm);
        // Only one numeric simple value corresponds to an aggregate indicator
        jiv.setMultiValue(false);
        return jiv;

    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        source.stopPolling();
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        source.startPolling();
    }

}
