/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */

package org.ow2.jasmine.probe.collectors.correlate.internal;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.util.MetaData;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

/**
 * Correlate Collector Implementation
 * @author durieuxp
 */
public class CorrelateCollector extends JCollector {

    private int ope;

    private Collection<JasmineCollector> sources;

    /**
     * Constructor
     * @param key
     * @param indic
     * @param period
     * @param ope
     * @param sources
     */
    public CorrelateCollector(String key, JasmineIndicator indic, int period, int ope, Collection<JasmineCollector> sources) {
        super(key, indic, period);
        this.ope = ope;
        this.sources = sources;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());

        HashMap<String, String> jsrProps = new HashMap<String, String>();
        Collection<JasmineSingleResult> jsrList = new ArrayList<JasmineSingleResult>();
        String domain = null;
        String server = null;
        String target = null;
        String mbean = null;
        long timestamp = 0;
        for (JasmineCollector elem : sources) {
            JasmineIndicatorValue jivs = elem.getLastResult();
            logger.debug("Get value of source indicator {0} for correlate indicator {1} ", elem.getIndicatorName(), indicator.getName());
            if (jivs == null) {
                logger.warn("Cannot obtain result for indicator {0} because no available value for source indicator {1} (null)", this.getIndicatorName(), elem.getIndicatorName());
                return null;
            }
            if (timestamp == 0) {
                timestamp = jivs.getTimestamp();
            } else {
                if (jivs.getTimestamp() > timestamp) {
                    // keep the biggest
                    timestamp = jivs.getTimestamp();
                }
            }
            if (jivs.getValues().isEmpty()) {
                logger.warn("Cannot obtain result for indicator {0} because no available value for source indicator {1} (values is empty)", this.getIndicatorName(), elem.getIndicatorName());
                return null;
            }
            if (jivs.getValues().size() > 1) {
                logger.warn("Correlation on multivalued indicator not supported");
                throw new JasmineCollectorException("Correlation on multivalued indicator not supported");
            }
            // Get the JSR (the only one)
            for (JasmineSingleResult jsr1 : jivs.getValues()) {
                jsrList.add(jsr1);
                HashMap<String, String> jsr1Props = jsr1.getProperties();
                for (String key : jsr1Props.keySet()) {
                    if (key.equals(MetaData.MBEAN)) {
                        String mbeanValue = jsr1Props.get(key);
                        if (mbean == null) {
                            mbean = mbeanValue;
                        } else {
                            if (!mbean.equals(mbeanValue)) {
                                mbean = MetaData.MISC_CORRELATE_MBEAN;
                            }
                        }
                    } else {
                        jsrProps.put(key, jsr1Props.get(key));
                    }
                }
                break;
            }
            // Analyse the JIV's target and meatadata domain and server
            String d = jivs.getMetadata().get(MetaData.DOMAIN);
            if (d != null) {
                if (domain == null) {
                    domain = d;
                } else if (!d.equals(domain)) {
                    domain = MetaData.MISC_DOMAIN;
                }
            }
            String s = jivs.getMetadata().get(MetaData.SERVER);
            if (s != null) {
                if (server == null) {
                    server = s;
                } else if (!s.equals(server)) {
                    server = MetaData.MISC_SERVER;
                }
            }
            String t = jivs.getTarget();
            if (t != null) {
                if (target == null) {
                    target = t;
                } else if (!target.equals(t)) {
                    if (!NO_TARGET.equals(t)) {
                        if (target.equals(NO_TARGET)) {
                            // replace NO_TARGET by t
                            target = t;
                        } else {
                            // 2 or more different targets
                            target = MISC_TARGET;
                        }
                    } // else, ignore NO_TARGET, keep target as already set
                }
            }
        }
        jiv.addMetadata(MetaData.DOMAIN, domain);
        jiv.addMetadata(MetaData.SERVER, server);
        jiv.setTarget(target);
        if (jsrList.isEmpty()) {
            logger.warn("No result available for correlate indicator {0}", indicator.getName());
            return null;
        }
        Number result = 0;
        switch (ope) {
            case CorrelateCollectorService.OP_ADD:
                result = addJsr(jsrList);
                break;
            case CorrelateCollectorService.OP_SUB:
                result = diffJsr(jsrList);
                break;
            case CorrelateCollectorService.OP_MUL:
                result = multiplyJsr(jsrList);
                break;
            case CorrelateCollectorService.OP_DIV:
                result = divideJsr(jsrList);
                break;
            case CorrelateCollectorService.OP_PERCENT:
                Number div = divideJsr(jsrList);
                result = 100 * div.doubleValue();
                break;
            default:
                throw new JasmineCollectorException("Bad correlate operation");
        }
        if (indicator.getScale() != 1) {
            result = divideValues(result, indicator.getScale());
        }
        JasmineSingleResult jsr = new JasmineSingleNumberResult();
        jsr.setTimestamp(timestamp);
        jsr.setValue(result);
        jsr.setName("");
        jsrProps.put(MetaData.MBEAN, mbean);
        jsr.setProperties(jsrProps);
        jiv.addValue(jsr);
        // Currently only one numeric simple value corresponds to a correlate indicator
        // as correlation on multiValued indicator not supported
        jiv.setMultiValue(false);
        return jiv;
    }

    /**
     * Stop polling
     */
    @Override
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        for (JasmineCollector source : sources) {
            source.stopPolling();
        }
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        for (JasmineCollector source : sources) {
            source.startPolling();
        }
    }

    // ------------------------------------------------------------------------------------
    // private methods
    // ------------------------------------------------------------------------------------

}
