/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.TabularData;

public class ComplexValuesUtil {


    /**
     * Return an element in a TabularData value corresponding to a given index.
     * @param value the TabularData value
     * @param index the given index
     * @return the corresponding element if the index is valid
     * @throws ComplexValuesException could not access to the TabularData or the value
     * has multiple indexes.
     */
    public static Object getTabElement(Object value, String index) throws ComplexValuesException {
        Object element = null;
        TabularData tabData = (TabularData) value;
        Collection<?> values = tabData.values();
        for (Object tabValue : values) {
            // the rows in the TabularData
            CompositeData compositeValue = (CompositeData) tabValue;
            try {
                Object[] tabIndexes = tabData.calculateIndex(compositeValue);
                if (tabIndexes.length == 1) {
                    String tabIndex = (String) tabIndexes[0];
                    if (tabIndex.equals(index)) {
                        // found corresponding row
                        element = tabValue;
                        break;
                    }
                } else {
                    // Support for multi-indexed tabs
                    // TODO
                    throw new ComplexValuesException("Complex values having multi-index TabularType are not yet supported");
                }
            } catch (Exception e) {
                throw new ComplexValuesException("Exception when trying to access a TabularData's element with index: " + index);
            }
        }
        return element;
    }

    /**
     * Return an array's element corresponding to a given index.
     * @param value the Array value
     * @param index the given index
     * @return the corresponding element if the index is valid
     * @throws ComplexValuesException the index has not a number
     */
    public static Object getArrayElement(Object value, String index) throws ComplexValuesException, ClassCastException {
        Object element = null;
        try {
            int i = Integer.parseInt(index);
            element = Array.get(value, i);
        } catch (NumberFormatException nfe) {
            throw new ComplexValuesException("Element has unsupported index format:" + index);
            // index is not a number !
        } catch (IndexOutOfBoundsException e) {
            throw new ComplexValuesException("Cannot access array element with out of range index: " + index);
        }
        return element;
    }

    /**
     * Return a List element corresponding to a given index.
     * @param value the List value
     * @param index the given index
     * @return the corresponding element if the index is valid
     * @throws ComplexValuesException the index has not a number
     */
    public static Object getListElement(Object value, String index) throws ComplexValuesException {
        Object element = null;
        try {
            int i = Integer.parseInt(index);
            List<?> lvalue = (List<?>) value;
            element = lvalue.get(i);
        } catch (NumberFormatException nfe) {
            throw new ComplexValuesException("Element has unsupported index format:" + index.toString());
            // index is not a number !
        } catch (IndexOutOfBoundsException e) {
            throw new ComplexValuesException("Cannot access list element with out of range index: " + index);
        }
        return element;
    }

    /**
     * Get an item of a CompositeData value
     * @param value the CompositeData value
     * @param key the given key
     * @return the item corresponding to the key, if the key exists
     * @throws ComplexValuesException no item found with the given key
     */
    public static Object getCompositeItem(Object value, String key) throws ComplexValuesException {
        Object item;
        CompositeData compositeValue = (CompositeData) value;
        Set<String> keys = compositeValue.getCompositeType().keySet();
        for (String _key : keys) {
            if (key.equals(_key)) {
                item = compositeValue.get(key);
                return item;
            }
        }
        // not found item with the given key
        throw new ComplexValuesException("CompositeData value has no key equal to " + key);
    }

    /**
     * Get an item of a Map value
     * @param value the Map value
     * @param key the given key
     * @return the item corresponding to the key, if the key exists
     * @throws ComplexValuesException no item found with the given key
     */
    public static Object getMapItem(Object value, String key) throws ComplexValuesException {
        Object item;
        Map<?, ?> mapValue = (Map<?, ?>) value;
        Set<?> keys = mapValue.keySet();
        for (Object _key : keys) {
            try {
                String sKey = (String) _key;
                if (key.equals(sKey)) {
                    item = mapValue.get(sKey);
                    return item;
                }
            } catch (ClassCastException e) {
                throw new ComplexValuesException("Complex object has a Map type with non String keys !");
            }
        }
        // not found item with the given key
        throw new ComplexValuesException("Map type value has no key equal to " + key);
    }

    /**
     * Get an element of an indexed complexValue
     * @param value the complex value
     * @return the element if found (null should not be returned)
     * @throws ComplexValuesException not found (could be the case for a TablularType) or could not access.
     */
    public static Object getElement(Object value, String index) throws ComplexValuesException {
        if (value instanceof TabularData) {
            Object element = getTabElement(value, index);
            if (element == null) {
                throw new ComplexValuesException("Cannot access Tabular element with inexistent index: " + index);
            }
            return element;
        } else if (value instanceof List<?>) {
            return getListElement(value, index);
        } else {
            try {
                return getArrayElement(value, index);
            } catch (ClassCastException cce) {
                throw new ComplexValuesException("Indexed elements other then TabularType, List, Array or Object[] are not supported");
            }
        }
    }

    /**
     * Get an item of a complexValue
     * @return
     * @throws ComplexValuesException
     */
    public static Object getItem(Object value, String key) throws ComplexValuesException {
        Object item = null;
        boolean found = false;
        if (value instanceof CompositeData) {
            return getCompositeItem(value, key);
        } else if (value instanceof Map<?, ?>) {
            return getMapItem(value, key);
        } else {
            throw new ComplexValuesException("Complex object has neither a Map type value, nor a CompositeData value !");
        }
    }

    /**
     * This method allows to define the simple types allowed by JasmineProbe.
     * Simple types are not decomposed in elements !
     * @param value value to test
     * @return true is its value is simple, false otherwise.
     */
    public static boolean hasSimpleType(Object value) {
        if (value instanceof Number
                || value instanceof String
                || value instanceof ObjectName
                || value instanceof Boolean) {
            return true;
        } else {
            return false;
        }
    }
}
