/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.jmx.JmxBrowserService;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;


/**
 * This Collector Service creates Jmx Collectors.
 *
 * @author durieuxp
 */
@Component(name = "JmxCollectorService")
@Provides
public class JmxCollectorService extends JCollectorService implements JmxBrowserService {

    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Get monitoring data from MBeans registered in an MBean server.";
    public final static String LONG_DESC =
            "MBean name (or filter) should be conform to the MBean ObjectName syntax.\nOne or more attributes or fragments can be specified.";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "No source indicator used.";
    public final static int SOURCE_MIN = 0;
    public final static int SOURCE_MAX = 0;

    /**
     * Properties that can be set when defining a "jmx" indicator
     */
    public final static String PROP_TARGET = "target";
    public final static String PROP_TARGET_DESC =
            "JMX target name. If not specified, a target must be specified by a probe using this indicator.";
    public final static String PROP_MBEAN = "mbean";
    public final static String PROP_MBEAN_DESC =
            "OBJECT_NAME or OBJECT_NAME pattern. Allows to determine the MBean(s) to be polled.";
    public final static String PROP_ATTR = "attr";
    public final static String PROP_ATTR_DESC =
            "Comma separated list of attributes or attribute fragment names. If not specified, all attribute of the mbean.";

    public final static String PROP_TEST = "test";

    public final static String DEFAULT_SERVER = "unknown";
    public final static String DEFAULT_DOMAIN = "unknown_domain";

    /**
     * List of Workers indexed by target names.
     */
    private HashMap<String, JmxWorker> workerList = new HashMap<String, JmxWorker>();

    /**
     * The value of the PROP_ATTR property meaning that all the
     * attributes are to be fetched.
     */
    public final static String VALUE_ALL = "all";

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name = "indicator.type", value = "jmx")
    private String indicatorType;

    @Requires
    private JmxConnectionService jmxConnectionService = null;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(0);
        collectorInfo.setSourceMax(0);
        // init the list of indicator properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_TARGET, PROP_TARGET_DESC, false));
        properties.add(new JasminePropertyInfo(PROP_MBEAN, PROP_MBEAN_DESC, true));
        properties.add(new JasminePropertyInfo(PROP_ATTR, PROP_ATTR_DESC, false));
        collectorInfo.setPropertyInfos(properties);
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
        // Stop workers
        for (JmxWorker worker : workerList.values()) {
            worker.remove();
        }
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for a given JasmineIndicator used by a JasmineProbe.
     * If a collector is already registered in the <code>collectorList</code>
     * using a key equal to indicatorName + probeName, it will be simply returned.
     * Moreover, if the JasmineProbe specifies a target that is different from
     * the one of the current worker, change the worker to the one on the specified target.
     * @param indicator a JasmineIndicator
     * @param probe a JasmineProbe
     * @return the Collector corresponding to the given indicator and probe
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe) throws JasmineCollectorException {

        // sanity check, not really needed.
        if (!indicator.getType().equals(indicatorType)) {
            logger.error("Bad type: " + indicator.getType());
            throw new JasmineCollectorException("Bad type: " + indicator.getType());
        }

        // Get probe's target (currently only one target accepted in a probe definition)
        String probeTarget = null;
        if (!probe.getTargetList().isEmpty()) {
            probeTarget = probe.getTargetList().get(0);
        }

        // Create collector
        // Get properties
        Map<String, String> props = indicator.getProperties();

        // Get Jmx specifics parameters
        String mbean = props.get(PROP_MBEAN);
        String attr = props.get(PROP_ATTR);
        String test = props.get(PROP_TEST);

        // Get JasmineTarget
        String targetName = null;
        if (probeTarget != null) {
            targetName = probeTarget;
        } else {
            targetName = props.get(PROP_TARGET);
        }
        if (targetName == null) {
            logger.error("Undefined target for probe {0} and indicator {1} ", probe.getId(), indicator.getName());
            throw new JasmineCollectorException("Undefined target for probe " + probe.getId() + " and indicator " + indicator.getName());
        }

        JasmineTarget target = probeManager.getTarget(targetName);
        if (target == null) {
            logger.error("Unknown target: " + target);
            throw new JasmineCollectorException("Unknown target: " + targetName);
        }

        // check target definition
        if (target.getProperties().isEmpty()) {
            logger.error("Wrong definition for target: {0}. Properties are missing.", targetName);
            throw new JasmineCollectorException("Wrong definition for target: " + targetName);
        } else {
            // check 'url' property
            if (target.getProperties().get("url") == null) {
                logger.error("Wrong definition for target: {0}. Property url is missing.", targetName);
                throw new JasmineCollectorException("Wrong definition for target: " + targetName);
            }
        }

        JCollector collector = null;
        if ("true".equals(test)) {
            // create a test purpose JmxCollector
            JCollector coll = getCollector(indicator.getName(), probe.getId());
            if (coll == null) {
                collector = new JmxCollectorSimple(probe.getId(), indicator, probe.getPeriod(), mbean, attr);
                ((JmxCollectorSimple) collector).setJmxConnectionService(jmxConnectionService);
                ((JmxCollectorSimple) collector).setTarget(target);
                addCollector(indicator.getName(), collector);
            } else {
                if (coll instanceof JmxCollectorSimple) {
                    JmxCollectorSimple collThread = (JmxCollectorSimple) coll;
                    ++(collThread.nb);
                    collector = collThread;
                }
            }


        } else {
            // create a regular JmxCollector
            collector = new JmxCollector(probe.getId(), indicator, probe.getPeriod(), mbean, attr);

            // Get the worker
            JmxWorker worker = null;
            if (workerList.containsKey(targetName)) {
                // TODO check that the target doesn't changed ...
                worker = workerList.get(targetName);
            } else {
                worker = new JmxWorker(target, jmxConnectionService);
                workerList.put(targetName, worker);
            }
            worker.addCollector((JmxCollector) collector);

            addCollector(indicator.getName(), collector);
        }


        return collector;

    }

    /**
     * Returns the List of targets used by this indicator.
     * Overloads JCollectorService.getDependantTargets()
     * @param indic JasmineIndicator
     * @return Returns the List of targets used by this indicator.
     */
    public List<String> getDependantTargets(JasmineIndicator indic) {
        List<String> ret = new ArrayList<String>();
        Map<String, String> props = indic.getProperties();
        for (String key : props.keySet()) {
            if (PROP_TARGET.equals(key)) {
                // get the target name
                ret.add(props.get(key));
            }
        }
        return ret;
    }

    /**
     * Removing a target implies to remove the corresponding workers.
     */
    public synchronized void removeTarget(String targetName) {
        JmxWorker workerToremove = workerList.get(targetName);
        if (workerToremove != null) {
            workerToremove.remove();
            workerList.remove(targetName);
        }
    }

}
