/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.JasmineSingleBooleanResult;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleObjectNameResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.JasmineSingleStringResult;
import org.ow2.jasmine.probe.api.generated.JsrType;
import org.ow2.jasmine.probe.api.generated.PropType;
import org.ow2.jasmine.probe.api.generated.TypeNameType;
import org.ow2.jasmine.probe.api.generated.ValueType;
import org.ow2.jasmine.probe.rest.Task;
import org.ow2.jasmine.probe.rest.Tasks;

/**
 * Task objects are used to keep data collected by probes (JasmineProbeResult)
 * and transform this data into XML format. The transformed data is used by the
 * JTasks singleton object that implements the GET /task REST commands.
 * Data is set by dedicated "rest" typed outputs.
 *
 * @author danesa
 */
public class JTask implements Task {

    private String probeId = null;
    private String taskId = null;

    private JasmineProbeResult lastResult = null;

    public JTask(String probeId, String taskId) {
        this.probeId = probeId;
        this.taskId = taskId;
    }

    /**
     * Used by the JTasks singleton to get data in XML format.
     * @return the JasmineProbeResult transformed into XML data
     */
    protected List<JAXBElement<JsrType>> getLastResult() {
        if (lastResult == null) {
            return null;
        }

        List<JAXBElement<JsrType>> elementsList = new ArrayList<JAXBElement<JsrType>>();

        //String probeName = lastResult.getProbeId();

        for (JasmineIndicatorValue jiv : lastResult.getValues()) {
            List<PropType> metaList = new ArrayList<PropType>();
            Map<String, String> indicMeta = jiv.getMetadata();
            for (String key : indicMeta.keySet()) {
                String val = indicMeta.get(key);
                PropType prop = new PropType();
                prop.setKey(key);
                prop.setValue(val);
                metaList.add(prop);
            }
            for (JasmineSingleResult jsr : jiv.getValues()) {
                JsrType jsrElement = new JsrType();
                String name = composeName(jiv, jsr.getName());
                jsrElement.setName(name);
                // Continue to set jsrElement's attributes ...
                //
                jsrElement.setTimestamp(jsr.getTimestamp());
                Map<String, String> resProps = jsr.getProperties();
                List<PropType> propList = new ArrayList<PropType>(metaList);
                for (String key : resProps.keySet()) {
                    String val = resProps.get(key);
                    PropType prop = new PropType();
                    prop.setKey(key);
                    prop.setValue(val);
                    propList.add(prop);
                }
                jsrElement.setProperty(propList);

                ValueType value = new ValueType();
                TypeNameType type = getValueAndType(jsr, value);
                jsrElement.setType(type);
                jsrElement.setValue(value);

                QName elementName  = new QName("org.ow2.jasmine.probe:probe-config", "data");
                JAXBElement<JsrType> xmlElement =  new JAXBElement<JsrType>(elementName, JsrType.class, jsrElement);
                elementsList.add(xmlElement);
            }
        }
        return elementsList;
    }

    /**
     * Used by the a dedicated "rest" output to set the probe's result into the task.
     */
    public void setLastResult(JasmineProbeResult result) {
        Tasks.logger.debug("Last result for probe {0} set into task {1}", probeId, taskId);
        this.lastResult = result;
    }

    @Override
    public String getTaskId() {
        return taskId;
    }

    private TypeNameType getValueAndType(final JasmineSingleResult jsr, ValueType value) {
        if (jsr instanceof JasmineSingleNumberResult) {
            JasmineSingleNumberResult numberRes = (JasmineSingleNumberResult) jsr;
            Number numvalue = numberRes.getValue();
            try {
                Integer intvalue = (Integer) numvalue;
                value.setIntvalue(intvalue.intValue());
                return TypeNameType.INT;
            } catch (ClassCastException e) {
                // try other type
            }
            try {
                Short shortvalue = (Short) numvalue;
                value.setShortvalue(shortvalue.shortValue());
                return TypeNameType.SHORT;
            } catch (ClassCastException e) {
                // try other type
            }
            try {
                Long longvalue = (Long) numvalue;
                value.setLongvalue(longvalue.longValue());
                return TypeNameType.LONG;
            } catch (ClassCastException e) {
                // try other type
            }
            try {
                Float floatvalue = (Float) numvalue;
                value.setFloatvalue(floatvalue.floatValue());
                return TypeNameType.FLOAT;
            } catch (ClassCastException e) {
                // try other type
            }
            try {
                Double doublevalue = (Double) numvalue;
                value.setDoublevalue(doublevalue.doubleValue());
                return TypeNameType.DOUBLE;
            } catch (ClassCastException e) {
                // try other type
            }
        }
        if (jsr instanceof JasmineSingleBooleanResult) {
            JasmineSingleBooleanResult booleanRes = (JasmineSingleBooleanResult) jsr;
            Boolean booleanvalue = (Boolean) booleanRes.getValue();
            value.setBooleanvalue(booleanvalue);
            return TypeNameType.BOOLEAN;
        }
        if (jsr instanceof JasmineSingleStringResult) {
            JasmineSingleStringResult stringRes = (JasmineSingleStringResult) jsr;
            String stringvalue = stringRes.getValue();
            value.setStringvalue(stringvalue);
            return TypeNameType.STRING;
        }
        if (jsr instanceof JasmineSingleObjectNameResult) {
            JasmineSingleObjectNameResult onRes = (JasmineSingleObjectNameResult) jsr;
            ObjectName onvalue = (ObjectName) onRes.getValue();
            value.setStringvalue(onvalue.toString());
            return TypeNameType.OBJECT_NAME;
        }
        value = null;
        return TypeNameType.UNKNOWN;
    }

    /**
     * Compose name of value as in JOuter !!
     * ***** TODO move this code to a common module ****
     * @param jiv
     * @param jsrName name of the JasmineSingleResult
     * @return Name of the value
     */
    public String composeName(JasmineIndicatorValue jiv, String jsrName) {
        StringBuffer name = new StringBuffer(jiv.getName());
        if (jiv.isMultiValue()) {
            if (jsrName != null && jsrName.length() != 0) {
                name.append(".");
                name.append(jsrName);
            }
        }
        return name.toString();
    }
}
