/**
 * JASMINe
 * Copyright (C) 2013 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

package org.ow2.jasmine.probe.shell;

import org.apache.felix.gogo.commands.Action;
import org.apache.felix.gogo.commands.Argument;
import org.apache.felix.gogo.commands.Command;
import org.apache.felix.gogo.commands.Option;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.HandlerDeclaration;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.service.command.CommandSession;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.List;
import java.util.Map;

/**
 * Shell command: indicator-change
 *
 * @author durieuxp
 */
@Component
@Command(name = "indicator-change",
        scope = "jasmine",
        description = "Change a JASMINe indicator")
@HandlerDeclaration("<sh:command xmlns:sh='org.ow2.shelbie'/>")
public class IndicatorChange implements Action {

    @Requires
    JasmineProbeManager probeManager = null;

    @Argument(index = 0, name = "name", description = "Indicator Name", required = true, multiValued = false)
    private String name;

    @Option(name = "-t", aliases = "--type", description = "Indicator Type", required = false, multiValued = false)
    private String type;

    @Option(name = "-d", aliases = "--define", description = "Specific indicator properties given in key=value format", required = false, multiValued = true)
    private List<String> params;

    protected static Log logger = LogFactory.getLog(Util.class);

    public Object execute(CommandSession commandSession) {

        logger.debug("");

        // Retrieve the old indicator definition
        JasmineIndicator old = null;
        try {
            old = probeManager.getIndicator(name);
        } catch (Exception e1) {
            logger.error("There is no JasmineIndicator named " + name);
            return null;
        }

        try {
            // Keep old definition when not redefined
            if (type == null) {
                type = old.getType();
            }
            Map<String, String> properties = old.getProperties();

            if (params != null) {
                for (String param : params) {
                    if (Util.validParam(param)) {
                        int indexEqual = param.indexOf("=");
                        String paramName = param.substring(0, indexEqual);
                        String paramValue = param.substring(indexEqual + 1, param.length());
                        properties.put(paramName, paramValue);
                    } else {
                        logger.error("indicator-change " + name + " failed");
                        logger.error("Wrong properties format for " + param);
                        logger.error("Expected format is paramName=paramValue");
                        return null;
                    }
                }
            }

            // create new JasmineIndicator
            JasmineIndicator n = new JasmineIndicator();
            n.setName(name);
            n.setType(type);
            n.setProperties(properties);

            logger.debug(n.toString());
            probeManager.changeIndicator(n);
            logger.info("Changing " + name + " successful");
        } catch (Exception e) {
            logger.error("indicator-change failed", e);
        }
        return null;
    }

}