/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.util.internal;


import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularType;

/**
 * This Value Object holds the result of a collect for an indicator.
 */
public class Metric {

    /**
     * Metric time stamp
     */
    private long timestamp = 0;

    /**
     * ObjectName of the MBean
     */
    private ObjectName mbean = null;

    /**
     * Indicator name
     */
    private String name = null;

    /**
     * Indicator value
     */
    private Object value = null;

    /**
     * Indicator type
     */
    private String type = null;

    /**
     * Constructor
     */
    public Metric() {
    }

    public ObjectName getMbean() {
        return mbean;
    }

    public void setMbean(ObjectName mbean) {
        this.mbean = mbean;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Object getValue() {
        return value;
    }

    public void setValue(Object value) {
        this.value = value;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public ObjectName getMBean() {
        return mbean;
    }

    public String toString() {
        return "* MBean " + mbean.toString() + ", Name " + name + ", Type " + type + ", Value = " + value;
    }

    /**
     * Create a Metric instance.
     * @param mbean Source MBean
     * @param name metric name
     * @param timestamp when the data was read
     * @param type the type of the data (a simple type)
     * @param value the value of the data
     * @return a Metric instance.
     */
    private static Metric getMetric(ObjectName mbean, String name, long timestamp, String type, Object value) {
        Metric metric = new Metric();
        metric.setMbean(mbean);
        metric.setName(name);
        metric.setTimestamp(timestamp);
        metric.setType(type);
        metric.setValue(value);
        return metric;
    }


    /**
     * Try to create a Metric instance.
     * @param mbean Source MBean
     * @param name metric name if the data has a simple type. Otherwise, the name is the path constructed staring with the attribute name,
     * running through the fragments hierarchy down to the current complex type fragment.
     * @param timestamp when the data was read
     * @param data the data that was read from the MBean
     * @return the metric if the data has a simple type, null otherwise
     */
    public static Metric getMetric(ObjectName mbean, String name, long timestamp, Object data) {
        Metric metric = null;
        if (data instanceof java.lang.Integer) {
            Integer value = (Integer) data;
            metric = getMetric(mbean, name, timestamp, "java.lang.Integer", value);
        } else if (data instanceof java.lang.Long) {
            Long value = (Long) data;
            metric = getMetric(mbean, name, timestamp, "java.lang.Long", value);
        } else if (data instanceof java.lang.String) {
            String value = (String) data;
            metric = getMetric(mbean, name, timestamp, "java.lang.String", value);
        }  else if (data instanceof java.lang.Boolean) {
            Boolean value = (Boolean) data;
            metric = getMetric(mbean, name, timestamp, "java.lang.Boolean", value);
        } else {
            // TODO
            // Complete with other simple types
        }
        return metric;
    }

    /**
     * Return the Metrics corresponding to a data having a supported non simple type.
     * Otherwise return null.
     * @param mbean Source MBean
     * @param name metric name if the data has a simple type. Otherwise, the name is the path constructed staring with the attribute name,
     * running through the fragments hierarchy down to the current complex type fragment.
     * @param timestamp when the data was read
     * @param data the data that was read from the MBean
     * @return the list of the metrics corresponding to the fragments of the data. If the data has a simple type,
     * the list is composed by one metric. If the data has a type that is not supported, null is returned.
     */
    public static List<Metric> getMetrics(ObjectName mbean, String name, long timestamp, Object data) {
        Metric metric = getMetric(mbean, name, timestamp, data);
        if (metric != null) {
            List<Metric> metrics = new ArrayList<Metric>();
            metrics.add(metric);
            return metrics;
        }
        // complex type
        if (data.getClass().isArray()) {
            return getArrayMetrics(mbean, name, timestamp, data);
        } else if (data instanceof CompositeData) {
            return getCompositDataMetrics(mbean, null, name, timestamp, data);
        } else if (data instanceof TabularData) {
            return getTabularDataMetrics(mbean, name, timestamp, data);
        } else {
            System.out.println("To Do: determine metrics if a complex value that's not an Array nor a CompositeData or TabularData !!");
        }
        return null;
    }


    /**
     * Return the Metrics corresponding to an array data.
     * @param mbean Source MBean
     * @param name path to the array data
     * @param timestamp when the data was read
     * @param data the data that was read from the MBean
     * @return
     */
    private static List<Metric> getArrayMetrics(ObjectName mbean, String name, long time, Object data) {
        ArrayList<Metric> metrics = new ArrayList<Metric>();
        int length = Array.getLength(data);
        for (int index = 0; index < length; index++) {
            Object element = Array.get(data, index);
            String elementName = FragmentName.getArrayElementName(name, index);
            Metric elementMetric = getMetric(mbean, elementName, time, element);
            if (elementMetric != null) {
                metrics.add(elementMetric);
            } else {
                List<Metric> elementMetrics = getMetrics(mbean, elementName, time, element);
                metrics.addAll(elementMetrics);
            }
        }
        return metrics;

    }

    /**
     * Return the Metrics corresponding to some of the items of a CompositeData.
     * @param data attribute or fragment having a CompositeData type
     * @param mbean Source MBean
     * @param name path to the composite data.
     * @param time when the data was read
     * @return
     */
    private static List<Metric> getCompositDataMetrics(ObjectName mbean, List<String> indexItemName, String name, long time, Object data) {
        ArrayList<Metric> metrics = new ArrayList<Metric>();
        CompositeData compData = (CompositeData) data;
        CompositeType type = compData.getCompositeType();
        Set<String> keySet = type.keySet();
        Iterator<String> keyIt = keySet.iterator();
        while (keyIt.hasNext()) {
            String key = keyIt.next();
            if ((indexItemName != null) && indexItemName.contains(key)) {
                // this key is an index, we don't need to generate metrics for it
                System.out.println("ingnored key = " + key);
                continue;
            }
            Object item = compData.get(key);
            String compName = FragmentName.getCompName(name, key);
            Metric componentMetric = Metric.getMetric(mbean, compName, time, item);
            if (componentMetric != null) {
                metrics.add(componentMetric);
                System.out.println(componentMetric.toString());
            } else {
                List<Metric> componentMetrics = getMetrics(mbean, compName, time, item);
                for (Metric subComponentMetric : componentMetrics) {
                    metrics.add(subComponentMetric);
                    System.out.println(subComponentMetric.toString());
                }
            }
        }
        return metrics;
    }

    /**
     * Return the Metrics corresponding to a TabularData
     * @param data tabularData to get metrics from it
     * @param mbean corresponding MBean
     * @param name the name path of the tabular data
     * @param time access time
     * @return the Metric objects corresponding to the data
     */
    private static List<Metric> getTabularDataMetrics(ObjectName mbean, String name, long time, Object data) {
        ArrayList<Metric> metrics = new ArrayList<Metric>();
        // data object to run through
        TabularData tabData = (TabularData) data;

        TabularType tabType = tabData.getTabularType();
        List<String> indexNames = tabType.getIndexNames();
        for (String indexName : indexNames) {
            System.out.println("index name : " + indexName);
        }

        Collection<?> values = tabData.values();

        for (Object value : values) {
            // the rows in the TabularData (may be simple or complex)
            CompositeData compValue = (CompositeData) value;
            Object[] indexes = tabData.calculateIndex(compValue);
            CompositeData row = tabData.get(indexes);

            if (row.equals(compValue)) {
                System.out.println("OK test for tabData.get(indexes) !!!");
            }

            String rowName = FragmentName.getRowName(name, indexes);

            List<Metric> rowMetrics = getCompositDataMetrics(mbean, indexNames, rowName, time, compValue);

        }
        return metrics;
    }
}
