/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineProbeManager.java 8658 2011-07-26 08:59:52Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe;

import java.util.Collection;

/**
 * Interface for JASMINe Probe Manager.
 * @author durieuxp
 */
public interface JasmineProbeManager {

    // -------------------------------------------------------------------------
    // probes
    // -------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    String createProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Change a Probe with new parameters
     * @param probe probe description
     */
    void changeProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Remove a probe.
     * @param id probe identifier.
     */
    void removeProbe(final String id) throws JasmineProbeException;

    /**
     * Remove all the managed probes.
     * @throws JasmineProbeException
     */
    void removeAllProbes() throws JasmineProbeException;

    /**
     * @return all the probe definitions
     */
    Collection<JasmineProbe> getProbes();

    /**
     * Start a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    void startProbe(final String id) throws JasmineProbeException;

    /**
     * Stop a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    void stopProbe(final String id) throws JasmineProbeException;

    /**
     * Start all the managed probes.
     * @throws JasmineProbeException
     */
    void startAllProbes() throws JasmineProbeException;

    /**
     * Stop all the managed probes.
     * @throws JasmineProbeException
     */
    void stopAllProbes() throws JasmineProbeException;

    // -------------------------------------------------------------------------
    // outputs
    // -------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param output output description
     * @return the Id to be used to reference this output later.
     */
    String createOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Change an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param output output description
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void changeOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void removeOutput(final String name) throws JasmineProbeException;

    /**
     * get the list of all defined outputs.
     * @return List of the outputs
     */
    Collection<JasmineOutput> getOutputs();

    /**
     * get the list of possible properties for Output, depending of its type.
     * @param type type of the Output (console, file, ...)
     * @return Collection of JasminePropertyInfo
     */
    Collection<JasminePropertyInfo> getOutputPropertyInfos(String type);

    // -------------------------------------------------------------------------
    // indicators
    // -------------------------------------------------------------------------

    /**
     * Create a new Indicator
     * @param indic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    String createIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * Change an indicator. It must exist, otherwise an exception is thrown.
     * The indicator must not be used by a probe, otherwise an exception is thrown.
     * @param indic indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     * or if it is used by a running probe.
     */
    void changeIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * remove an indicator
     * @param name ident of the indicator to remove
     */
    void removeIndicator(final String name) throws JasmineProbeException;

    /**
     * get the list of all defined indicators.
     * @return List of the indicators
     */
    Collection<JasmineIndicator> getIndicators();

    /**
     * get the list of possible properties for Indicator, depending of its type.
     * @param type type of the Indicator (Jmx, Lewys, ...)
     * @return Collection of JasminePropertyInfo
     */
    Collection<JasminePropertyInfo> getIndicatorPropertyInfos(String type);

    // -------------------------------------------------------------------------
    // targets
    // -------------------------------------------------------------------------

    /**
     * Create a new target
     * @param target target description
     * @return the Id to be used to reference this target later.
     */
    String createTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     * @param target new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    void changeTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * remove a target
     * @param name ident of the target to remove
     */
    void removeTarget(final String name) throws JasmineProbeException;

    /**
     * get the list of all defined targets.
     * @return List of the targets
     */
    Collection<JasmineTarget> getTargets();

    // -------------------------------------------------------------------------
    // other operations
    // -------------------------------------------------------------------------

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     * @param listener object that treats the probe state change
     */
    void addProbeListener(JasmineProbeListener listener);

    /**
     * Save the current configuration in the specified xml file
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    void saveConfig(String path) throws JasmineProbeException;

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     * @param path the xml file, or null if load the default configuration file.
     * @throws JasmineProbeException
     */
    void loadConfig(String path) throws JasmineProbeException;

}

