/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbeManagerMXBean.java 8858 2011-09-13 14:30:11Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe;

import java.util.Collection;
import java.util.Set;

/**
 * MXBean for JASMINe Probe Manager.
 * MXBean interface can be in a different package than implementation,
 * and the implementation may have a different name.
 * @author durieuxp
 */
public interface JProbeManagerMXBean {

    // ------------------------------------------------------------------------
    // operations on indicators
    // ------------------------------------------------------------------------

    /**
     * Create a new Indicator
     * @param indic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    String createIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * Get the list of indicators
     */
    String [] listIndicators() throws JasmineProbeException;

    /**
     * Get an indicator by its name
     * @param id indicator identifier.
     * @return the Indicator definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineIndicator getIndicator(final String id) throws JasmineProbeException;

    /**
     * remove an indicator
     * @param name ident of the indicator to remove
     */
    void removeIndicator(final String name) throws JasmineProbeException;

    /**
     * @return the available indicator types.
     */
    Set<String> getIndicatorTypes();

    // ------------------------------------------------------------------------
    // operations on targets
    // ------------------------------------------------------------------------

    /**
     * Create a new target
     * @param target target description
     * @return the Id to be used to reference this target later.
     */
    String createTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Get the list of targets
     */
    String [] listTargets() throws JasmineProbeException;

    /**
     * Get a Target by its name
     * @param id target identifier.
     * @return the Target definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineTarget getTarget(final String id) throws JasmineProbeException;

    /**
     * remove a target
     * @param name ident of the target to remove
     */
    void removeTarget(final String name) throws JasmineProbeException;

    // ------------------------------------------------------------------------
    // operations on outputs
    // ------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param output output description
     * @return the Id to be used to reference this output later.
     */
    String createOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Get the list of outputs
     */
    String [] listOutputs() throws JasmineProbeException;

    /**
     * Get an Output by its name
     * @param id output identifier.
     * @return the Output definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineOutput getOutput(final String id) throws JasmineProbeException;

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void removeOutput(final String name) throws JasmineProbeException;

    /**
     * @return the available output types.
     */
    Set<String> getOutputTypes();

    // ------------------------------------------------------------------------
    // operations on probes
    // ------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    String createProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Start a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    void startProbe(final String id) throws JasmineProbeException;

    /**
     * Stop a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    void stopProbe(final String id) throws JasmineProbeException;

    /**
     * Start all the managed probes.
     * @throws JasmineProbeException
     */
    void startAllProbes() throws JasmineProbeException;

    /**
     * Stop all the managed probes.
     * @throws JasmineProbeException
     */
    void stopAllProbes() throws JasmineProbeException;

    /**
     * Remove a probe.
     * @param id probe identifier.
     */
    void removeProbe(final String id) throws JasmineProbeException;

    /**
     * Remove all the managed probes.
     * @throws JasmineProbeException
     */
    void removeAllProbes() throws JasmineProbeException;

    /**
     * Get the list of probes
     */
    String [] listProbes() throws JasmineProbeException;

    /**
     * Get a Probe by its name
     * @param id probe identifier.
     * @return the Probe definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineProbe getProbe(final String id) throws JasmineProbeException;

    // ------------------------------------------------------------------------
    // operations on configuration
    // ------------------------------------------------------------------------

    /**
     * Save the current configuration in the specified xml file
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    void saveConfig(String path) throws JasmineProbeException;

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     * @param path the xml file, or null if load the default configuration file.
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    void loadConfig(String path) throws JasmineProbeException;

}

