/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineProbeManager.java 8858 2011-09-13 14:30:11Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe;

import java.util.Collection;
import java.util.Set;

/**
 * Interface for JASMINe Probe Manager.
 * Most of operations are inherited as part of the MXBean interface.
 * @author durieuxp
 */
public interface JasmineProbeManager extends JProbeManagerMXBean {

    // -------------------------------------------------------------------------
    // probes
    // -------------------------------------------------------------------------

    /**
     * Change a Probe with new parameters
     * @param probe probe description
     */
    void changeProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * @return all the probe definitions
     */
    Collection<JasmineProbe> getProbes();

    // -------------------------------------------------------------------------
    // outputs
    // -------------------------------------------------------------------------

    /**
     * Change an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param output output description
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void changeOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * get the list of all defined outputs.
     * @return List of the outputs
     */
    Collection<JasmineOutput> getOutputs();

    /**
     * get the list of possible properties for Output, depending of its type.
     * @param type type of the Output (console, file, ...)
     * @return Collection of JasminePropertyInfo
     */
    Collection<JasminePropertyInfo> getOutputPropertyInfos(String type);

    // -------------------------------------------------------------------------
    // indicators
    // -------------------------------------------------------------------------

    /**
     * Change an indicator. It must exist, otherwise an exception is thrown.
     * The indicator must not be used by a probe, otherwise an exception is thrown.
     * @param indic indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     * or if it is used by a running probe.
     */
    void changeIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * get the list of all defined indicators.
     * @return List of the indicators
     */
    Collection<JasmineIndicator> getIndicators();

    /**
     * get the list of possible properties for Indicator, depending of its type.
     * @param type type of the Indicator (Jmx, Lewys, ...)
     * @return Collection of JasminePropertyInfo
     */
    Collection<JasminePropertyInfo> getIndicatorPropertyInfos(String type);


    // -------------------------------------------------------------------------
    // targets
    // -------------------------------------------------------------------------

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     * @param target new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    void changeTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * get the list of all defined targets.
     * @return List of the targets
     */
    Collection<JasmineTarget> getTargets();

    // -------------------------------------------------------------------------
    // other operations
    // -------------------------------------------------------------------------

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     * @param listener object that treats the probe state change
     */
    void addProbeListener(JasmineProbeListener listener);

}

