/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe;

import java.util.List;
import java.util.Set;

/**
 * MXBean for JASMINe Probe Manager.
 * MXBean interface can be in a different package than implementation,
 * and the implementation may have a different name.
 * @author durieuxp
 */
public interface JProbeManagerMXBean {

    // ------------------------------------------------------------------------
    // operations on indicators
    // ------------------------------------------------------------------------

    /**
     * Create a new Indicator
     * @param indic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    String createIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * Change an indicator. It must exist.
     * @param indic indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     * or if it is used by a running probe.
     */
    void changeIndicator(final JasmineIndicator indic) throws JasmineProbeException;

    /**
     * Get the list of all indicators or of indicators of a given type
     * @param type the given type. If null, get all the indicators.
     */
    String [] listIndicators(final String type);

    /**
     * Get an indicator by its name
     * @param id indicator identifier.
     * @return the Indicator definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineIndicator getIndicator(final String id) throws JasmineProbeException;

    /**
     * get the list of all defined indicators.
     * @return List of the indicators
     */
    List<JasmineIndicator> getIndicators();

    /**
     * remove an indicator
     * @param name ident of the indicator to remove
     */
    void removeIndicator(final String name) throws JasmineProbeException;

    /**
     * @return the available indicator types.
     */
    Set<String> getIndicatorTypes();

    /**
     * Get all infiormation about the type of indicator
     * @param type type of the Indicator (Jmx, Lewys, ...)
     * @return JasmineIndicatorTypeInfo
     */
    JasmineCollectorInfo getIndicatorTypeInfo(String type);

    /**
     * Get the list of probes using this indicator
     * @param name  Indicator name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    Set<String> getIndicatorCurrentUse(String name) throws JasmineProbeException;

    // ------------------------------------------------------------------------
    // operations on targets
    // ------------------------------------------------------------------------

    /**
     * Create a new target
     * @param target target description
     * @return the Id to be used to reference this target later.
     */
    String createTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     * @param target new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    void changeTarget(final JasmineTarget target) throws JasmineProbeException;

    /**
     * Get the list of targets
     */
    String [] listTargets() throws JasmineProbeException;

    /**
     * Get a Target by its name
     * @param id target identifier.
     * @return the Target definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineTarget getTarget(final String id) throws JasmineProbeException;

    /**
     * get the list of all defined targets.
     * @return List of the targets
     */
    List<JasmineTarget> getTargets();

    /**
     * remove a target
     * @param name ident of the target to remove
     */
    void removeTarget(final String name) throws JasmineProbeException;

    /**
     * @return the available target types.
     */
    Set<String> getTargetTypes();

    /**
     * get the list of possible properties for Target, depending of its type.
     * @param type type of the Target (today: only jmx is supported)
     * @return List of JasminePropertyInfo
     */
    List<JasminePropertyInfo> getTargetPropertyInfos(String type);

    // ------------------------------------------------------------------------
    // operations on outputs
    // ------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param output output description
     * @return the Id to be used to reference this output later.
     */
    String createOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Change an output. The output must exist.
     * @param output output description
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void changeOutput(final JasmineOutput output) throws JasmineProbeException;

    /**
     * Get the list of all outputs or of outputs having a given type
     * @param type the given type. If null, get all the outputs.
     */
    String [] listOutputs(final String type);

    /**
     * Get the list of the default outputs.
     */
    String [] listDefaultOutputs();

    /**
     * Add an output to the default outputs list.
     * @param id output identifier
     * @throws JasmineProbeException If there is no output with the given identifier
     */
    void addDefaultOutput(final String id) throws JasmineProbeException;

    /**
     * Remove an output from the default outputs list.
     * @param id output identifier
     * @throws JasmineProbeException If there is no output with the given identifier in the default outputs list.
     */
    void removeDefaultOutput(final String id) throws JasmineProbeException;

    /**
     * Get an Output by its name
     * @param id output identifier.
     * @return the Output definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineOutput getOutput(final String id) throws JasmineProbeException;

    /**
     * get the list of all defined outputs.
     * @return List of the outputs
     */
    List<JasmineOutput> getOutputs();

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     * or if it is used by a running probe.
     */
    void removeOutput(final String name) throws JasmineProbeException;

    /**
     * @return the available output types.
     */
    Set<String> getOutputTypes();

    /**
     * get the list of possible properties for Output, depending of its type.
     * @param type type of the Output (console, file, ...)
     * @return List of JasminePropertyInfo
     */
    List<JasminePropertyInfo> getOutputPropertyInfos(String type);

    /**
     * Get the list of probes using this output
     * @param name  Output name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    Set<String> getOutputCurrentUse(String name) throws JasmineProbeException;

    // ------------------------------------------------------------------------
    // operations on probes
    // ------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    String createProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Change a Probe with new parameters
     * @param probe probe description
     */
    void changeProbe(final JasmineProbe probe) throws JasmineProbeException;

    /**
     * Set a new value for probe period
     * @param id probe identifier.
     * @param period in seconds
     */
    void changeProbePeriod(final String id, final int period) throws JasmineProbeException;

    /**
     * Start a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    void startProbe(final String id) throws JasmineProbeException;

    /**
     * Stop a probe.
     * @param id probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    void stopProbe(final String id) throws JasmineProbeException;

    /**
     * Start all the managed probes.
     * @throws JasmineProbeException
     */
    void startAllProbes() throws JasmineProbeException;

    /**
     * Stop all the managed probes.
     * @throws JasmineProbeException
     */
    void stopAllProbes() throws JasmineProbeException;

    /**
     * Remove a probe.
     * @param id probe identifier.
     */
    void removeProbe(final String id) throws JasmineProbeException;

    /**
     * Remove all the managed probes.
     * @throws JasmineProbeException
     */
    void removeAllProbes() throws JasmineProbeException;

    /**
     * Get the list of probes
     */
    String [] listProbes() throws JasmineProbeException;

    /**
     * Get a Probe by its name
     * @param id probe identifier.
     * @return the Probe definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    JasmineProbe getProbe(final String id) throws JasmineProbeException;

    /**
     * @return all the probe definitions
     */
    List<JasmineProbe> getProbes();

    // ------------------------------------------------------------------------
    // operations on configuration
    // ------------------------------------------------------------------------

    /**
     * Save the current configuration in the specified xml file
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    void saveConfig(String path) throws JasmineProbeException;

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     * @param path the xml file, or null if load the default configuration file.
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    void loadConfig(String path) throws JasmineProbeException;

}

