/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A JasmineIndicator defines how to obtain the value to be monitored
 * @author durieuxp
 */
public class JasmineIndicator {

    /**
     * Name of this indicator
     * can be provided by the user or generated.
     * Must be unique inside of an agent
     */
    protected String name;

    /**
     * type of Indicator: should identify the CollectorService to be used.
     */
    private String type = null;

    /**
     * Scale factor: The rough result will be divided by this value.
     * Default is 1.
     */
    private int scale = 1;

    /**
     * List of Indicators used to build this Indicator.
     * May be an empty list.
     */
    private List<String> sources = new ArrayList<String>();

    /**
     * Type specific parameters
     */
    private Map<String, String> properties = new HashMap<String, String>();

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public int getScale() {
        return scale;
    }

    public void setScale(int scale) {
        this.scale = scale;
    }

    public List<String> getSources() {
        return sources;
    }

    public void setSources(List<String> sources) {
        this.sources = sources;
    }

    public Map<String, String> getProperties() {
        return properties;
    }

    public void setProperties(Map<String, String> properties) {
        this.properties = properties;
    }

    /**
     * Build a full description of this object.
     * @return
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("JasmineIndicator " + name);
        buffer.append("\n> type : " + type);
        buffer.append("\n> scale : " + scale);
        buffer.append("\n> sources : " + sources);
        for (String key : properties.keySet()) {
            buffer.append("\n> " + key + " = " + properties.get(key));
        }
        return buffer.toString();
    }

    /**
     * Build a single line description of this object.
     * @return
     */
    public String toLine()  {
        String formattedName = (name + "                              ").substring(0, 30);
        return formattedName + type;
    }

    /**
     * Compare the current JasmineIndicator with a given one end return true
     * only if name, type, scale, sources and properties are identical.
     * @param other the JasmineIndicator to compare this object with
     * @return true if name, type, scale, sources and properties are identical,
     * false otherwise.
     */
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (! (obj instanceof JasmineIndicator)) {
            return false;
        }
        JasmineIndicator other = (JasmineIndicator) obj;
        if (!name.equals(other.getName())) {
            return false;
        }
        if (!type.equals(other.getType())) {
            return false;
        }
        if (scale != other.getScale()) {
            return false;
        }
        if (sources.size() != other.sources.size()) {
            return false;
        }
        for (String source : sources) {
            if (!other.sources.contains(source)) {
                return false;
            }
        }
        if (properties.size() != other.properties.size()) {
            return false;
        }
        for (String key : properties.keySet()) {
            if (!other.properties.containsKey(key)) {
                return false;
            }
            String value = properties.get(key);
            if (value != null) {
                if (!value.equals(other.properties.get(key))) {
                    return false;
                }
            } else {
                if (other.properties.get(key) != null) {
                    return false;
                }
            }
        }
        return true;
    }
}
