/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DerivedCollectorService.java 8599 2011-07-21 08:20:27Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.derived.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Manages a list of JCollector objects.
 * @author durieuxp
 */
@Component(name="DerivedCollectorService")
@Provides
public class DerivedCollectorService implements JasmineCollectorService {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * List of collectors
     */
    private Map<String, DerivedCollector> collectorList = new HashMap<String, DerivedCollector>();

    /**
     * Info about Properties
     */
    private List<JasminePropertyInfo> properties;

    /**
     * Property names
     */
    public final static String PROP_OP = "op";
    public final static String PROP_SOURCE = "source";

    /**
     * List of supported operations
     */
    public final static int OP_PREV = 1;
    public final static int OP_DELTA = 2;
    public final static int OP_RATE = 3;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="derived")
    private String indicatorType;

    @Validate
    public void start() {
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_OP, "derived operation (delta|prev|rate)", true));
        properties.add(new JasminePropertyInfo(PROP_SOURCE, "source indicator name", true));
    }

    @Invalidate
    public void stop() {
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe
     * @param indicator
     * @param period
     * @param probeid
     * @return the JasmineCollector
     * @throws JasmineCollectorException
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid)
            throws JasmineCollectorException {

        // Build a unique key from the probeid and the indicator name
        String key = probeid + ":" + indicator.getName();

        DerivedCollector collector = collectorList.get(key);
        if (collector == null) {
            // sanity check, not really needed.
            if (! indicator.getType().equals(indicatorType)) {
                logger.error("Bad type: " + indicator.getType());
                throw new JasmineCollectorException("Bad type: " + indicator.getType());
            }

            // Get the collector corresponding to the source indicator
            Map<String, String> props = indicator.getProperties();
            String iname = props.get(PROP_SOURCE);
            JasmineCollector source = null;
            try {
                source = probeManager.getCollector(iname, period, probeid);
            } catch (JasmineProbeException e) {
                logger.error("Cannot get value: " + e);
                throw new JasmineCollectorException(e.getMessage());
            }

            // Check the operation
            String operation = props.get(PROP_OP);
            if (operation == null) {
                String err = "Operation not defined in DerivedCollectorService";
                logger.error(err);
                throw new JasmineCollectorException(err);
            }
            // Create the Collector
            collector = new DerivedCollector(key, indicator, period, operation, source);
            collectorList.put(key, collector);
        }
        return collector;
    }


    /**
     * Retrieve the description of all specific properties hidden in JasmineOutput
     * Return a Map of all properties with their description
     * @return  a Map of property descriptions
     */
    public List<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }
}
