/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DerivedCollector.java 9039 2011-10-03 12:22:06Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.derived.internal;

import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;

/**
 * Derived collector implementation.
 * @author durieuxp
 */
public class DerivedCollector extends JCollector {

    /**
     * operation code
     */
    private int ope;
    private String operation;

    private JasmineCollector source;

    private JasmineIndicatorValue previous = null;

    /**
     * Constructor
     * @param name identify the collector
     * @param indic indicator definition VO
     * @param period execution period
     * @param op derived operation
     * @param source collector that provides the initial values.
     */
    public DerivedCollector(String name, JasmineIndicator indic, int period, String op, JasmineCollector source) {
        super(name, indic, period);
        this.operation = op;
        if (operation.equalsIgnoreCase("prev")) {
            ope = DerivedCollectorService.OP_PREV;
        } else if (operation.equalsIgnoreCase("delta")) {
            ope = DerivedCollectorService.OP_DELTA;
        } else if (operation.equalsIgnoreCase("rate")) {
            ope = DerivedCollectorService.OP_RATE;
        } else {
            String err = "Operation not supported in DerivedCollectorService: " + operation;
            logger.error(err);
            return;
        }

        this.source = source;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        JasmineIndicatorValue current = source.getLastResult();

        // First call: No previous value yet.
        if (previous == null) {
            previous = current;
            logger.debug("First call: No previous value yet");
            return null;
        }

        long curtime = current.getTimestamp();
        long timeslice = curtime - previous.getTimestamp();

        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());
        jiv.addMetadata("op", operation);

        // Keep almost all metadata from source indicator (server, domain, ...)
        for (String key : current.getMetadata().keySet()) {
            if (!key.equals("op")) {
                 jiv.addMetadata(key, current.getMetadata().get(key));
            }
        }
        jiv.setTarget(current.getTarget());

        for (JasmineSingleResult res : current.getValues()) {
            JasmineSingleResult old = previous.getValue(res.getName());
            JasmineSingleResult val = new JasmineSingleNumberResult();

            // Build a name for this indicator
            String name = indicator.getName();
            if (current.getValues().size() > 1) {
                name = name + "(" + res.getName() + ")";
            }
            val.setName(name);

            // take the timestamp of the current value
            val.setTimestamp(curtime);

            // take the properties of the current value
            // TODO verify that this is ok for all the supported source types
            HashMap<String, String> resProps = res.getProperties();
            if (!resProps.isEmpty()) {
                for (Iterator<String> propKeys = resProps.keySet().iterator(); propKeys.hasNext(); ) {
                    String propKey = propKeys.next();
                    String propVal = resProps.get(propKey);
                    val.addProperty(propKey, propVal);
                    logger.warn("Add source value property {0} to the derived value {1}", propKey, name);
                }
            }

            // Compute the value, depending on the operation
            Number value = null;
            switch (ope) {
                case DerivedCollectorService.OP_PREV:
                    // prev: Just take the old values
                    value = (Number) old.getValue();
                    if (indicator.getScale() != 1) {
                        value =  divideValues(value, indicator.getScale());
                    }
                    val.setValue(value);
                    break;
                case DerivedCollectorService.OP_DELTA:
                    // delta: Compute the difference between new and old values
                    value = diffValues((Number) res.getValue(), (Number) old.getValue());
                    if (indicator.getScale() != 1) {
                        value =  divideValues(value, indicator.getScale());
                    }
                    val.setValue(value);
                    break;
                case DerivedCollectorService.OP_RATE:
                    // rate: Difference divided by period in seconds
                    long diff = diffValues((Number) res.getValue(), (Number) old.getValue());
                    //float rate = TimeUnit.MILLISECONDS.toSeconds(diff) / timeslice;
                    float rate = diff / TimeUnit.MILLISECONDS.toSeconds(timeslice);
                    if (indicator.getScale() != 1) {
                        rate = rate / indicator.getScale();
                    }
                    val.setValue(rate);
                    break;
                default:
                    throw new JasmineCollectorException("Unimplemented derived operation " + ope);
            }
            jiv.addValue(val);
        }

        // return the computed value, and keep the new one as previous value.
        previous = current;
        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        // Just reinit previous value
        previous = null;
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("");
        // nothing to do
    }
}
