/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DerivedCollector.java 9680 2012-01-27 12:49:56Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.derived.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.TimeUnit;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;

/**
 * Derived collector implementation.
 * @author durieuxp
 * @author danesa
 */
public class DerivedCollector extends JCollector {

    /**
     * operation code
     */
    private int ope;
    private String operation;

    private JasmineCollector source;

    private JasmineIndicatorValue previous = null;

    /**
     * Constructor
     * @param probeId probe identifier
     * @param indic indicator definition VO
     * @param period execution period
     * @param op derived operation
     * @param source collector that provides the initial values.
     */
    public DerivedCollector(String probeId, JasmineIndicator indic, int period, String op, JasmineCollector source) {
        super(probeId, indic, period);
        this.operation = op;
        if (operation.equalsIgnoreCase("prev")) {
            ope = DerivedCollectorService.OP_PREV;
        } else if (operation.equalsIgnoreCase("delta")) {
            ope = DerivedCollectorService.OP_DELTA;
        } else if (operation.equalsIgnoreCase("rate")) {
            ope = DerivedCollectorService.OP_RATE;
        } else {
            String err = "Operation not supported in DerivedCollectorService: " + operation;
            logger.error(err);
            return;
        }

        this.source = source;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        // Current source value
        JasmineIndicatorValue current = source.getLastResult();
        if (current == null) {
            logger.warn("No result available on source indicator {0}", source.getIndicatorName());
            return null;
        }

        // Initialize previous value.
        if (previous == null) {
            previous = current;
            logger.warn("First call: No previous value yet for indicator {0} in probe {1}", getIndicatorName(), probeId);
            return null;
        }

        // Construct value to return.
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());

        // Keep all metadata from source indicator (server, domain, ...)
        for (String key : current.getMetadata().keySet()) {
             jiv.addMetadata(key, current.getMetadata().get(key));
        }
        jiv.setTarget(current.getTarget());
        jiv.setMultiValue(current.isMultiValue());

        for (JasmineSingleResult currentResult : current.getValues()) {
            JasmineSingleResult newRes = getDerivedResult(currentResult);
            newRes.setName(currentResult.getName());
            jiv.addValue(newRes);
        }

        // Keep the current source value as previous value.
        previous = current;
        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        source.stopPolling();
        // Reinit previous value
        previous = null;
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        source.startPolling();
    }

    /**
     * Construct a derived result from a current result provided by the source collector and the corresponding previous result.
     * The current result and its corresponding previous result have the same name.
     * @param currentResult a result provided by the source collector
     * @return the derived result
     * @throws JasmineCollectorException could not compute the derived result
     */
    private JasmineSingleResult getDerivedResult(final JasmineSingleResult currentResult) throws JasmineCollectorException {
        logger.debug("Compute derived result for {0} of indicator {1}", currentResult.getName(), getIndicatorName());

        JasmineSingleResult result = new JasmineSingleNumberResult();
        // Use the timestamp from the current result
        result.setTimestamp(currentResult.getTimestamp());
        // Use the properties from the current result
        HashMap<String, String> resProps = currentResult.getProperties();
        if (!resProps.isEmpty()) {
            for (Iterator<String> propKeys = resProps.keySet().iterator(); propKeys.hasNext(); ) {
                String propKey = propKeys.next();
                String propVal = resProps.get(propKey);
                result.addProperty(propKey, propVal);
                logger.debug("Add source result property {0} to the derived result {1}", propKey, result.getName());
            }
        }
        // Get the previous value.
        JasmineSingleResult previousResult = getPreviousResult(currentResult);
        if (previousResult == null) {
            result.setValue(null);
            return result;
        }
        // Compute result value from current value and previous value
        Number value = null;
        switch (ope) {
            case DerivedCollectorService.OP_PREV:
                // prev: Just take the previous value
                value = (Number) previousResult.getValue();
                break;
            case DerivedCollectorService.OP_DELTA:
                // delta: Compute the difference between current and previous values
                value = diffValues((Number) currentResult.getValue(), (Number) previousResult.getValue());
                logger.debug("Compute delta of {0} - {1}", currentResult.getValue(), previousResult.getValue());
                break;
            case DerivedCollectorService.OP_RATE:
                // rate: Difference divided by period in seconds
                Number diff = diffValues((Number) currentResult.getValue(), (Number) previousResult.getValue());
                long timeslice = currentResult.getTimestamp() - previousResult.getTimestamp();
                long timeSliceInSeconds = TimeUnit.MILLISECONDS.toSeconds(timeslice);
                value = divideValues(diff, timeSliceInSeconds).doubleValue();
                logger.debug("Compute rate for current value {0} and previous value {1} and timeslice {2}. Result is {3}", currentResult.getValue(), previousResult.getValue(), timeSliceInSeconds, value);
                break;
            default:
                throw new JasmineCollectorException("Unimplemented derived operation " + ope);
        }
        if (indicator.getScale() != 1) {
            value =  divideValues(value, indicator.getScale());
        }
        result.setValue(value);
        return result;
    }

    /**
     * Get the previous value of a given result
     * @param currentResult the given result
     * @return the corresponding previous result
     * @throws JasmineCollectorException
     */
    private JasmineSingleResult getPreviousResult(final JasmineSingleResult currentResult) throws JasmineCollectorException {
        ArrayList<JasmineSingleResult> possiblePrevResults =  new ArrayList<JasmineSingleResult>();
        for (JasmineSingleResult prevResult : previous.getValues()) {
            if (prevResult.getName().equals(currentResult.getName())) {
                possiblePrevResults.add(prevResult);
            }
        }
        if (possiblePrevResults.size() == 1) {
            // found one corresponding result
            logger.debug("Directly found previous value of indicator {0} result {1} ", getIndicatorName(), currentResult.getName());
            return possiblePrevResults.get(0);
        } else {
            // found the corresponding previous result based on metadata
            HashMap<String, String> resProps = currentResult.getProperties();
            if (resProps.isEmpty()) {
                throw new JasmineCollectorException("Could not found the corresponding previous value for result " + currentResult.getName() + " in indicator " + getIndicatorName());
            }
            for (JasmineSingleResult prevResult : possiblePrevResults) {
                boolean okPrevResult = true;
                HashMap<String, String> prevResultProps = prevResult.getProperties();
                for (Iterator<String> propKeys = resProps.keySet().iterator(); propKeys.hasNext(); ) {
                    String propKey = propKeys.next();
                    if (prevResultProps.containsKey(propKey)) {
                        // Compare property values
                        String propVal = resProps.get(propKey);
                        String prevPropVal = prevResultProps.get(propKey);
                        if (!propVal.equals(prevPropVal)) {
                            okPrevResult = false;
                        }
                    }
                }
                if (okPrevResult) {
                    // found the corresponding previous result
                    logger.debug("Found previous value of indicator {0} result {1} based on metadata matching", getIndicatorName(), currentResult.getName());
                    logger.debug("Prev result: ", prevResult.toString());
                    logger.debug("Current result: ", currentResult.toString());
                    return prevResult;
                }
            }
            throw new JasmineCollectorException("Could not found the corresponding previous value for result " + currentResult.getName() + " in indicator " + getIndicatorName());
        }
    }
}
