/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.derived.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * Implements a service that manages <code>derived</code> collectors.
 * @author durieuxp
 */
@Component(name="DerivedCollectorService")
@Provides
public class DerivedCollectorService extends JCollectorService {

    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Get a new value from a given source indicator's values.";
    public final static String LONG_DESC =
            "Several derived operations exist:"
          + "\nprev: return the value corresponding to the previous polling iteration"
          + "\ndelta: current(v) - previous(v), where v is the source indicator's value"
          + "\nrate: delta(v)/delta(t), where delta(t) is the time elapsed between the current and the previous polling iteration"
          ;

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "A source indicator must be provided. Its value can be composed of several results (each result should be a number).";
    public final static int SOURCE_MIN = 1;
    public final static int SOURCE_MAX = 1;

    /**
     * Properties
     */
    public final static String PROP_OP = "op";
    public final static String PROP_OP_DESC =
            "Derived operation. Must be one of : prev|delta|rate";

    /**
     * List of supported operations
     */
    public final static int OP_PREV = 1;
    public final static int OP_DELTA = 2;
    public final static int OP_RATE = 3;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="derived")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(SOURCE_MIN);
        collectorInfo.setSourceMax(SOURCE_MAX);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_OP, PROP_OP_DESC, true));
        collectorInfo.setPropertyInfos(properties);
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe.
     * First parse indicator properties, and get a collector for each source.
     * The goal is to 'notify' to sources that an additional 'consumer' collector exists for them.
     * This allows for collectors that manage a cache of values to update cache size.
     * Then create a collector instance with default name, if this is the first call to getCollector().
     * Otherwise, a new name is generated by the service before creating an additional instance.
     * @param indicator the indicator for which values are collected by the collector
     * @param probe the probe using the indicator
     * @return the JasmineCollector instance created to collect values for the given indicator
     * within the given probe.
     * @throws JasmineCollectorException The collector could not be created.
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe)
            throws JasmineCollectorException {

        // sanity check, not really needed.
        if (! indicator.getType().equals(indicatorType)) {
            logger.error("Bad type: " + indicator.getType());
            throw new JasmineCollectorException("Bad type: " + indicator.getType());
        }

        Map<String, String> props = indicator.getProperties();

        // Check the operation
        String operation = props.get(PROP_OP);
        if (operation == null) {
            String err = "Operation not defined in DerivedCollectorService";
            logger.error(err);
            throw new JasmineCollectorException(err);
        }

        // Get the collector corresponding to the source indicator
        if (indicator.getSources().isEmpty()) {
            logger.error("Derived indicator without source defined: " + indicator.getName());
            throw new JasmineCollectorException("Derived indicator without source defined: " + indicator.getName());
        }
        String iname = indicator.getSources().get(0);
        JasmineCollector source = null;
        try {
            source = probeManager.getCollector(iname, probe.getId());
        } catch (JasmineProbeException e) {
            logger.error("Cannot get value: " + e);
            throw new JasmineCollectorException(e.getMessage());
        }
        // Create the collector
        JCollector collector = new DerivedCollector(probe.getId(), indicator, probe.getPeriod(), operation, source);
        addCollector(indicator.getName(), collector);

        return collector;
    }

    /**
     * return the List of indicators used by this indicator.
     * @param indic JasmineIndicator
     * @return
     */
    @Override
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        return indic.getSources();
    }

}
