/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DfCollectorService.java 8743 2011-08-12 09:16:47Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.df.internal;

import org.apache.felix.ipojo.annotations.*;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.*;

/**
 * This CollectorService implements the "df" unix command.
 * Works only on linux like system, since the df command must exist.
 * @author durieuxp
 */
@Component(name="DfCollectorService")
@Provides
public class DfCollectorService implements JasmineCollectorService {

    /**
     * The name of the property allowing to get disk name.
     */
    static public final String PROP_DISK = "disk";

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * DiskFree thread collector
     * In this implementation, we have only one thread.
     */
    protected DiskFree df = null;

    /**
     * List of Collector
     */
    private HashMap<String, DfCollector> collectorList = new HashMap<String, DfCollector>();

    private Collection<JasminePropertyInfo> properties;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="df")
    private String indicatorType;

    @Validate
    public void start() {
        // init the list of indicator properties (needed for getPropertiesInfo)
        // these properties will be passed with the JasmineIndicator at getCollector()
        properties = new ArrayList<JasminePropertyInfo>();
        // If no disk specified: consider all the disks
        properties.add(new JasminePropertyInfo(PROP_DISK, "device to check (pathname relative to /dev). Example: sda1. When no specified: all the disks.", false));

        // Create the DiskFree ThreadCollector
        df = new DiskFree();
    }

    @Invalidate
    public void stop() {
        // Stop DiskFree
        df.remove();
        df = null;
    }

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for this JasmineIndicator.
     * Called by the JasmineProbeManager when first starting a probe
     * (at the probe creation).
     * @param indicator contains the indicator's parameters.
     * @param period in second
     * @param probeid the name of the probe that gets the collector
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid) throws JasmineCollectorException {
        // Build a key with the name and the probeid.
        String key = indicator.getName() + probeid;
        DfCollector collector = collectorList.get(key);
        if (collector == null) {
            // sanity check, not really needed
            if (! indicator.getType().equals(indicatorType)) {
                logger.error("Bad type: " + indicator.getType());
                throw new JasmineCollectorException("Bad type: " + indicator.getType());
            }
            // create a new DfCollector for this probe
            collector = new DfCollector(key, indicator, period);
            collectorList.put(key, collector);
            // Notify this new Collector to DiskFree
            df.addCollector(collector);
        }
        return collector;
    }

    /**
     * Retrieve the description of all specific properties hidden in JasmineCollector
     * Return a Collection of all properties with their description
     * @return  a Collection of property descriptions
     */
    public Collection<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }
}
