/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DfCollector.java 9017 2011-09-28 09:47:36Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.df.internal;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.util.CsvConverter;

import java.util.*;

/**
 * Collector implementation for df.
 * This implementation works with a separate thread used to pre collect the results.
 * A simpler implementation could be done by not using a thread.
 * Using a thread is interesting when we do not want to be stuck in case info
 * cannot be got.
 * @author durieuxp
 */
public class DfCollector extends JCollector {

    /**
     * Disk if specified, or null if no disk specified (= all disks)
     */
    private Collection<String> diskList = null;

    /**
     * List of values collected
     */
    private JasmineIndicatorValue cache = null;

    /**
     * Time to consider data as obsolete if not got by probe (in millisec).
     * TODO : make it configurable ...??
     */
    private static final long OBSOLATE_TIME = 3000L;

    /**
     * Pass stopped to true if the probe using this collector is stopped.
     */
    private boolean stopped = false;

    /**
     * Constructor
     * @param indicator definition
     */
    public DfCollector(String name, JasmineIndicator indicator, int period) {
        super(name, indicator, period);
        // specific part of JasmineIndicator (property list)
        Map<String, String> props = indicator.getProperties();
        this.diskList = CsvConverter.csv2list(props.get("disk"));
    }

    /**
     * Get the disk to be observed
     * null if we are interested on all disks
     * @return
     */
    public Collection<String> getDiskList() {
        return diskList;
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * This method return a List of results in case indicator represents
     * actually a list of value (for example: df for ALL the disks)
     * @return List of JasmineIndicatorValue
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        JasmineIndicatorValue jiv = cache;
        logger.debug("value=" + jiv);

        // Apply scale factor to the result, if not 1.
        if (indicator.getScale() != 1) {
            for (JasmineSingleResult jsr : jiv.getValues()) {
                long value = longValue((Number) jsr.getValue()) / indicator.getScale();
                jsr.setValue(value);
            }
        }

        // Set domain and server names in metadata
        // server is JmxCollectorService.META_SERVER = "server"
        jiv.addMetadata("server", getServerName());
        // domain is JmxCollectorService.META_DOMAIN;
        jiv.addMetadata("domain", getDomainName());
        // Set target in JIV
        if (hostName != null) {
            jiv.setTarget(hostName);
        } else {
            jiv.setTarget(getServerName());
        }
        // Clear the cache to avoid giving results twice.
        cache = null;

        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        stopped = true;
    }

    /**
     * Restart polling
     */
    public void startPolling() {
        logger.debug("");
        stopped = false;
    }
    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    /**
     * Check if this Collector needs a new result.
     * A stopped collector returns false.
     * Otherwise, it returns true if the cache is empty or outdated
     * regarding to the poll period and the OBSOLATE_TIME constant.
     * @return true if it needs one
     */
    public boolean needResult() {
        if (stopped) {
            return false;
        }
        if (cache == null) {
            return true;
        }
        long timestamp = cache.getTimestamp();
        if (oldvalue(timestamp)) {
            // forget this value too old.
            cache = null;
            return true;
        }
        return false;
    }

    /**
     * Add a new result to the list.
     * @param value
     */
    public void addResult(JasmineIndicatorValue value) {
        logger.debug(value.getName());
        cache = value;
    }

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * @return true if this timestamp is too old for the period.
     */
    private boolean oldvalue(long timestamp) {
        long now = System.currentTimeMillis();
        return (now - timestamp > period * 1000 + OBSOLATE_TIME);
    }
}
