/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DiskFree.java 9680 2012-01-27 12:49:56Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.df.internal;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.*;

/**
 * Give the disk free space in % for a given disk
 * This command uses the df command.
 * It works only on linux.
 * @author durieuxp
 */
public class DiskFree extends Thread {

    private Log logger = LogFactory.getLog(JCollector.class);

    private boolean started = false;

    /**
     * Period used to check for values to get.
     * TODO: Could be computed from period values
     */
    private long collectorTimeout = DEFAULT_TIMEOUT;
    private static final long DEFAULT_TIMEOUT = 1000; // 1 sec

    /**
     * List of Collectors managed by this Worker
     */
    private Collection<DfCollector> collectors = new ArrayList<DfCollector>();

    private HashMap<String, Integer> freespace = null;

    /**
     * Constructor
     */
    public DiskFree() {

    }

    /**
     * remove it
     */
    public synchronized void remove() {
        collectors.clear();
        started = false;
    }

    /**
     * Add a new Collector
     */
    public synchronized void addCollector(DfCollector col) {
        // Add the Collector to the list
        collectors.add(col);

        // Starts the Thread if not started yet.
        if (! started) {
            started = true;
            start();
        }
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (started) {
            // Check the collectors which were removed
            synchronized(this) {
                ArrayList<DfCollector> removed = new ArrayList<DfCollector>();
                for (DfCollector coll : collectors) {
                    if (coll.isRemoved()) {
                        removed.add(coll);
                    }
                }
                for (DfCollector coll : removed) {
                    collectors.remove(coll);
                }
            }
            // Check if must fetch some new results
            Collection<DfCollector> collupdate = new ArrayList<DfCollector>();
            synchronized(this) {
                for (DfCollector coll : collectors) {
                    if (coll.needResult()) {
                        collupdate.add(coll);
                    }
                }
            }

            if (! collupdate.isEmpty()) {
                // Keep timestamp identical for all these polling
                long time = System.currentTimeMillis();

                try {
                    // Get free space on all disks
                    // This is done outside the lock
                    getDiskUsage();

                    // Give results to the Collectors to update
                    synchronized(this) {
                        for (DfCollector coll : collupdate) {
                            String name = coll.getIndicator().getName();
                            JasmineIndicatorValue jiv = new JasmineIndicatorValue();
                            jiv.setName(name);
                            if (coll.getDiskList() == null) {
                                // We want info about ALL the disks
                                for (String disk : freespace.keySet()) {
                                    Integer val = freespace.get(disk);
                                    if (val == null) {
                                        logger.error("No value for " + disk);
                                    }
                                    // Build a result
                                    String jsrname = disk;
                                    JasmineSingleResult jsr = getJsr(jsrname, time, val);
                                    // Add this result to the JasmineIndicatorValue
                                    jiv.addValue(jsr);
                                }
                            } else {
                                // We are interested only with a set of disks
                                for (String disk : coll.getDiskList()) {
                                    Integer val = freespace.get(disk);
                                    if (val == null) {
                                        logger.error("No value for " + disk);
                                    }
                                    // Build a result
                                    String jsrname = disk;
                                    JasmineSingleResult jsr = getJsr(jsrname, time, val);
                                    // Add this result to the JasmineIndicatorValue
                                    jiv.addValue(jsr);
                                }
                                if (coll.getDiskList().size() ==1) {
                                    // one disk means one single simple result (JSR)
                                    jiv.setMultiValue(false);
                                }
                            }
                            coll.addResult(jiv);
                        }
                    }
                    collectorTimeout = DEFAULT_TIMEOUT;
                } catch (IOException e) {
                    logger.error("Cannot run df: " + e);
                    collectorTimeout = collectorTimeout * 2;
                } catch (InterruptedException e) {
                    logger.error("Cannot run df: " + e);
                    collectorTimeout = collectorTimeout * 2;
                }
            }

            // Wait
            synchronized(this) {
                try {
                    wait(collectorTimeout);
                } catch (InterruptedException e) {
                    logger.warn(getName() + ": collector interrupted", e);
                } catch (Exception e) {
                    logger.warn(getName() + ": collector exception", e);
                }
            }
        }
    }

    /**
     * Run the "df" unix command on all disks
     * @throws IOException
     * @throws InterruptedException
     */
    private void getDiskUsage() throws IOException, InterruptedException {
        freespace = new HashMap<String, Integer>();
        String cmd = "df -l";
        Runtime run = Runtime.getRuntime();
        Process pr = run.exec(cmd);
        pr.waitFor();
        BufferedReader buf = new BufferedReader(new InputStreamReader(pr.getInputStream()));
        while (true) {
            String line = buf.readLine();
            if (line == null) {
                break;
            }
            if (line.startsWith("/dev/")) {
                StringTokenizer stk = new StringTokenizer(line);
                String name = stk.nextToken().substring(5);
                Integer free = 0;
                while (stk.hasMoreTokens()) {
                    String tok = stk.nextToken();
                    int len = tok.length();
                    if (tok.endsWith("%") && len > 1) {
                        free = new Integer(tok.substring(0, len - 1));
                    }
                }
                freespace.put(name, free);
            }
        }
    }

    private JasmineSingleResult getJsr(final String name, final long time, final int value) {
        JasmineSingleResult jsr = new JasmineSingleNumberResult();
        jsr.setName(name);
        jsr.setTimestamp(time);
        jsr.setValue(value);
        return jsr;

    }
}
