/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxCollector.java 8645 2011-07-25 14:20:43Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import org.ow2.jasmine.probe.JasmineIndicValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.util.CsvConverter;

import javax.management.Attribute;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import java.lang.reflect.Array;
import java.util.*;

/**
 * Collector implementation for JMX.
 * This implementation works with a separate thread used to pre collect the results.
 * Using a thread is interesting when we do not want to be stuck in case info
 * cannot be got because the target is off.
 * @author durieuxp
 */
public class JmxCollector extends JCollector {

    /**
     * Jmx MBean Filter (ObjectName syntax)
     */
    private String pattern;
    private ObjectName objname = null;

    /**
     * Mbean Attributes to get.
     */
    private Collection<String> attrlist;

    /**
     * Last result got by JmxWorker
     */
    private List<Metric> metrics = null;

    /**
     * Last value returned, kept in case a probe need it twice.
     */
    private JasmineIndicValue cached = null;

    /**
     * Constructor
     * @param indicator definition
     */
    public JmxCollector(String name, JasmineIndicator indicator, int period, String mbean, String csv) {
        super(name, indicator, period);
        this.pattern = mbean;
        this.attrlist = CsvConverter.csv2list(csv);
        if (this.attrlist == null) {
            // TODO build list of all attributes
            // What if mbean is a pattern ?
        }
    }

    public ObjectName getObjectName() throws JasmineCollectorException {
        if (objname == null) {
            try {
                objname = new ObjectName(pattern);
            } catch (MalformedObjectNameException e) {
                throw new JasmineCollectorException("Malformed pattern: " + pattern);
            } catch (NullPointerException e) {
                throw new JasmineCollectorException("Null pattern");
            }
        }
        return objname;
    }

    /**
     * Get the list of attributes to fetch
     * @return
     */
    public String[] getAttributes() {
        String[] ret = new String[attrlist.size()];
        int i = 0;
        for (String elem : attrlist) {
            String attrname = attrName(elem);
            // TODO do not put attributes twice (if many composite are wanted)
            ret[i++] = attrname;
            logger.debug(attrname);
        }
        return ret;
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * This method return a List of results in case indicator represents
     * actually a list of value
     * @return JasmineIndicValue
     */
    @Override
    public JasmineIndicValue getLastResult() throws JasmineCollectorException {
        // First of all, return the cached value if any
        if (cached != null) {
            logger.debug("cached value returned");
            return cached;
        }
        if (metrics == null) {
            logger.warn("No result ready to return");
            return null;
        }
        JasmineIndicValue jiv = new JasmineIndicValue();
        String name = getIndicator().getName();
        logger.debug(name);
        jiv.setName(name);
        for (Metric metric : metrics) {
            if (metric.getAttributeList().size() == 0) {
                logger.warn("Result with no value: " + metric.toString());
                continue;
            }
            for (Iterator it = metric.getAttributeList().iterator(); it.hasNext(); ) {
                Attribute att = (Attribute) it.next();
                if (att.getValue() == null) {
                    // No value: forget this
                    logger.warn("no value for this indicator: " + att.getName());
                    continue;
                }
                // TODO metadata + properties

                // TODO Use lib for naming rules and process other types of attributes
                if (att.getValue().getClass().isArray()) {
                    logger.debug("array: " + att.getName());
                    // Array: build name_xx items
                    Array attval = (Array) att.getValue();
                    for (int i = 0; i < Array.getLength(attval); i++) {
                        String aname = att.getName() + "[" + i + "]";
                        JasmineSingleResult jsr = new JasmineSingleResult();
                        jsr.setName(aname);
                        jsr.setTimestamp(metric.getTimestamp());
                        Number value = (Number) Array.get(attval, i);
                        if (indicator.getScale() != 1) {
                            value = divideValues(value, indicator.getScale());
                        }
                        jsr.setValue(value);
                        jiv.addValue(jsr);
                    }
                } else if (att.getValue() instanceof Map) {
                    logger.debug("map: " + att.getName());
                    // Map: build name_key items
                    Map<?, ?> attval = (Map<?, ?>) att.getValue();
                    TreeSet<?> ts = new TreeSet<Object>(attval.keySet());
                    for (Iterator<?> itkeys = ts.iterator(); itkeys.hasNext(); ) {
                        Object key = itkeys.next();
                        String aname = att.getName() + "." + key.toString();
                        if (isWanted(aname)) {
                            JasmineSingleResult jsr = new JasmineSingleResult();
                            jsr.setName(aname);
                            jsr.setTimestamp(metric.getTimestamp());
                            Number value = (Number) attval.get(key);
                            if (indicator.getScale() != 1) {
                                value = divideValues(value, indicator.getScale());
                            }
                            jsr.setValue(value);
                            jiv.addValue(jsr);
                        }
                    }
                } else if (att.getValue() instanceof CompositeData) {
                    logger.debug("composite: " + att.getName());
                    // Composite: build name_fragment items
                    CompositeData attval = (CompositeData) att.getValue();
                    Set<String> ks = attval.getCompositeType().keySet();
                    for (Iterator<String> itkeys = ks.iterator(); itkeys.hasNext(); ) {
                        String key = (String) itkeys.next();
                        String aname = att.getName() + "." + key.toString();
                        if (isWanted(aname)) {
                            JasmineSingleResult jsr = new JasmineSingleResult();
                            jsr.setName(aname);
                            jsr.setTimestamp(metric.getTimestamp());
                            Number value = (Number) attval.get(key);
                            if (indicator.getScale() != 1) {
                                value = divideValues(value, indicator.getScale());
                            }
                            jsr.setValue(value);
                            jiv.addValue(jsr);
                        }
                    }
                } else {
                    // Simple attribute: keep it as is.
                    // Put name, target, mbean, values, timestamp inside JasmineIndicValue
                    String aname = att.getName();
                    logger.debug("simple: " + aname);
                    if (isWanted(aname)) {
                        JasmineSingleResult jsr = new JasmineSingleResult();
                        jsr.setName(aname);
                        jsr.setTimestamp(metric.getTimestamp());
                        Number value = (Number) att.getValue();
                        if (indicator.getScale() != 1) {
                            value = divideValues(value, indicator.getScale());
                        }
                        jsr.setValue(value);
                        jiv.addValue(jsr);
                    } else {
                        logger.warn("Attribute not wanted ? " + aname);
                    }
                }
            }
        }
        cached = jiv;
        metrics = null;
        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        // TODO
    }

    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    /**
     * Check if this Collector needs a new result
     * @return true if it needs one
     */
    public boolean needResult() {
        if (metrics == null) {
            return true;
        }
        long timestamp = metrics.get(0).getTimestamp();
        if (oldvalue(timestamp)) {
            return true;
        }
        return false;
    }

    /**
     * Add a new result to the list.
     * @param metrics
     */
    public void addResult(List<Metric> metrics) {
        if (metrics.size() == 0) {
            logger.warn("Empty metric list");
            return;
        }
        logger.debug("");
        this.metrics = metrics;
        this.cached = null;
        // TODO maybe better to compute cached now, instead of in getLastValue ?
    }

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * @return true if this timestamp is too old for the period.
     */
    private boolean oldvalue(long timestamp) {
        long now = System.currentTimeMillis();
        return (now - timestamp > period * 1000);
    }

    /**
     * Get the name of attribute, without composite extension.
     * TODO: tabular not taken in account.
     * @param elem
     * @return
     */
    private String attrName(String elem) {
        if (elem.endsWith("]")) {
            return elem.substring(0, elem.indexOf('['));
        }
        if (elem.indexOf('.') > 0) {
            return elem.substring(0, elem.indexOf('.'));
        }
        return elem;
    }

    /**
     * Check if an attribute is in the list of wanted attributes
     * @param attr
     * @return  true if wanted.
     */
    private boolean isWanted(String attr) {
        String root = attrName(attr);
        for (String elem : attrlist) {
            if (elem.equals(attr) || elem.equals(root)) {
                return true;
            }
        }
        return false;
    }
}
