/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxWorker.java 8595 2011-07-20 15:03:53Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import org.ow2.jasmine.probe.JasmineIndicValue;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionFactory;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import javax.management.AttributeList;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import java.util.*;

/**
 * Poll a Jmx target to get mbean attributes values.
 * @author durieuxp
 */
public class JmxWorker extends Thread {
    private Log logger = LogFactory.getLog(JCollector.class);

    private boolean started = false;

    private String targetname;
    private String url;

    /**
     * Period used to check for values to get.
     * TODO: Could be computed from period values
     */
    private long collectorTimeout = DEFAULT_TIMEOUT;
    private static final long DEFAULT_TIMEOUT = 1000; // 1 sec

    /**
     * List of Collectors managed by this Worker
     */
    private Collection<JmxCollector> collectors = new ArrayList<JmxCollector>();

    private JmxConnectionService jmxService;

    private Map<String,Object> env = new HashMap<String,Object>();

    private JmxConnectionFactory cfactory = null;

    /**
     * The name of the managed server represented by the target.
     */
    private String serverName = null;

    /**
     * The name of the management domain to which belongs the target server.
     */
    private String domainName = null;

    /**
     * Constructor
     */
    public JmxWorker(JasmineTarget target, JmxConnectionService jmxConnectionService) {
        logger.debug(target.getName());
        this.targetname = target.getName();
        this.url = target.getUrl();
        this.jmxService = jmxConnectionService;
    }

    /**
     * remove it
     */
    public synchronized void remove() {
        started = false;
    }

    /**
     * Add a new Collector
     */
    public synchronized void addCollector(JmxCollector col) {
        // Add the Collector to the list
        collectors.add(col);

        // Starts the Thread if not started yet.
        if (! started) {
            started = true;
            start();
        }
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (started) {
            // Check if must fetch some new results
            Collection<JmxCollector> topoll = new ArrayList<JmxCollector>();
            synchronized(this) {
                for (JmxCollector coll : collectors) {
                    if (coll.needResult()) {
                        topoll.add(coll);
                    }
                }
            }

            if (! topoll.isEmpty()) {
                // Keep timestamp identical for all these polling
                long t = System.currentTimeMillis();

                // Get a MBeanServerConnection
                MBeanServerConnection cnx = null;
                try {
                    cnx = getMBeanServerConnection();
                } catch (JmxConnectionException e) {
                    logger.warn("Cannot get Jmx Access:" + e);
                }
                if (cnx != null) {
                    if ((serverName == null) || (domainName == null)) {
                        setServerAndDomain(cnx);
                    }
                    // Fetch these results (outside the lock)
                    // this may block in case the target is off.
                    for (JmxCollector coll : topoll) {
                        List<Metric> metrics = collect(cnx, coll);
                        coll.addResult(metrics);
                    }
                }
                // release the MBeanServerConnection
                // nothing to do today...
            }

            // Wait
            synchronized(this) {
                try {
                    wait(collectorTimeout);
                } catch (InterruptedException e) {
                    logger.warn(getName() + ": collector interrupted", e);
                } catch (Exception e) {
                    logger.warn(getName() + ": collector exception", e);
                }
            }
        }
    }

    // -----------------------------------------------------------------------------------------
    // Private methods
    // -----------------------------------------------------------------------------------------

    /**
     * Get a MBeanServerConnection
     * @return MBeanServerConnection
     */
    private MBeanServerConnection getMBeanServerConnection() throws JmxConnectionException {
        if (cfactory == null) {
            cfactory = jmxService.getJmxConnectionFactory(url, env);
        }
        return cfactory.getMBeanServerConnection();
    }

    private void setServerAndDomain(MBeanServerConnection cnx) {
        try {
            ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
            Iterator<?> onames = cnx.queryNames(j2eeinstance, null).iterator();
            ObjectName serverOn = (ObjectName) onames.next();
            this.serverName = (String) cnx.getAttribute(serverOn, "serverName");
            this.domainName = (String) serverOn.getDomain();
        } catch (Exception e) {
            logger.warn("Cannot get server and domain names for target ", targetname);
        }

    }

    /**
     * Collect values of MBean attributes for this Collector
     * @param cnx MBeanServerConnection
     * @param col JmxCollector
     * @return
     */
    private List<Metric> collect(MBeanServerConnection cnx, JmxCollector col) {
        LinkedList<Metric> metrics = new LinkedList<Metric>();
        try {
            ObjectName objname = col.getObjectName();
            if (objname.isPattern()) {
                logger.debug("pattern:" + objname.toString());
                // Must loop on all the mbeans matching the pattern
                Set<ObjectName> onset = cnx.queryNames(objname, null);
                for (ObjectName mbean : onset) {
                    logger.debug("mbean:" + mbean.toString());
                    long time = System.currentTimeMillis();
                    AttributeList attl = cnx.getAttributes(mbean, col.getAttributes());
                    metrics.add(new Metric(time, mbean, attl, targetname));
                }
            } else {
                logger.debug("single mbean:" + objname.toString());
                long time = System.currentTimeMillis();
                AttributeList attl = cnx.getAttributes(objname, col.getAttributes());
                metrics.add(new Metric(time, objname, attl, targetname));
            }
        } catch (Exception e) {
            // TODO
            logger.warn("getAttributes error on JmxServer: " + e);
        }
        logger.debug("nb of metrics found: " + metrics.size());
        return metrics;
    }



}
