/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FragmentUtil.java 8752 2011-08-23 13:41:59Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.TabularData;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Manage JMX Attribute fragments.
 *
 * Attribute fragments are elements of complex attributes.
 *
 * Sometimes, users need to poll such attribute elements, for example an item in a Map,
 * and don't need the entire attribute value.
 * A naming policy is necessary to define the attributes elements to be polled by a JMX indicator.
 *
 * Also, when a JMX indicator specifies an attribute (or an attribute fragment) whose value has a complex
 * type, the value must be decomposed in simple elements (elementary fragments).
 * The naming policy is used to identify the fragments returned by a probe polling this indicator.
 *
 * @author danesa
 */
public class FragmentUtil {

    private static Log logger = LogFactory.getLog(JmxCollector.class);

    /**
     * Decompose a JMX Attribute having a complex type into a list of Attributes having
     * simple types (the fragments).
     * The supported complex types are:
     * <ul>
     *   <li>javax.management.openmbean.CompositeData</li>
     *   <li>javax.management.openmbean.TabularData</li>
     *   <li>Array</li>
     *   <li>String[]</li>
     *   <li>Map</li>
     * </ul>
     * The supported simple types are:
     * <ul>
     *   <li>Number</li>
     *   <li>Boolean</li>
     *   <li>String</li>
     * </ul>
     *
     * @param att The initial attribute which may have a complex or a simple type.
     * It its value has a simple type, the attribute is returned as it is.
     * If it has a complex type, its value is decomposed in simple elements, the fragments.
     *
     * @return A list of fragments. A fragment is contained in an Attribute instance.
     * A fragment's value is a simple value obtained by a complex value decomposition.
     * A fragment's name is a name is based on the fragments naming policy.
     * It starts with the initial attribute's name and contains a representation of the path
     * followed by the decomposition process.
     */
    public static List<Attribute> getFragments(Attribute att) {
        String name = att.getName();
        logger.debug(name);
        List<Attribute> result = new ArrayList<Attribute>();
        Object value = att.getValue();
        if (value instanceof Number
                || value instanceof String
                || value instanceof Boolean) {
            result.add(att);
            logger.debug("Return final fragment with name: " + name + " and value: " + value);
            return result;
        }
        if (value instanceof CompositeData) {
            CompositeData compValue = (CompositeData) value;
            Set<String> keys = compValue.getCompositeType().keySet();
            for (String key : keys) {
                Attribute itemAtt = getCompositeItemAttribute(name, key, compValue);
                List<Attribute> frags = getFragments(itemAtt);
                result.addAll(frags);
            }
            return result;
        }
        if (value instanceof TabularData) {
            TabularData tabData = (TabularData) value;

            Collection<?> values = tabData.values();

            for (Object tabValue : values) {
                // the rows in the TabularData (may be simple or complex)
                CompositeData compValue = (CompositeData) tabValue;
                Object[] indexes = tabData.calculateIndex(compValue);
                Attribute tabAtt = getTabularDataAttribute(name, indexes, compValue);
                List<Attribute> frags = getFragments(tabAtt);
                result.addAll(frags);
            }
            return result;
        }
        if (value.getClass().isArray()) {
            try {
                Array attval = (Array) value;
                for (int i = 0; i < Array.getLength(attval); i++) {
                    Attribute elemAtt = getArrayElementAttribute(name, i, attval);
                    List<Attribute> frags = getFragments(elemAtt);
                    result.addAll(frags);
                }
                return result;
            } catch (ClassCastException cc) {
                String[] attval = (String[]) value;
                for (int i = 0; i < attval.length; i++) {
                    String elem = attval[i];
                    String elemName = getNameForArrayElement(name, (new Integer(i)).toString());
                    Attribute elemAtt = new Attribute(elemName, elem);
                    List<Attribute> frags = getFragments(elemAtt);
                    result.addAll(frags);
                }
                return result;
            }
        }
        if (value instanceof Map) {
            Map<?, ?> attval = (Map<?, ?>) att.getValue();
            Set<?> keys = attval.keySet();
            for (Object key : keys) {
                Attribute elemAtt = getMapElementAttribute(name, key, attval);
                List<Attribute> frags = getFragments(elemAtt);
                result.addAll(frags);
            }
            return result;
        }

        logger.debug("No attribute returned, the type is not supported: {0}", value.getClass().getName());
        return result;
    }

    /**
     * Construct a fragment who's value is an element of an array.
     * @param name The starting name: attribute name or fragment name having as value the array.
     * Its a fragment name, if the array has been obtained by decomposing an encompassing value.
     * @param index the element's index
     * @param attval the array containing the element's value
     * @return the created fragment
     */
    private static Attribute getArrayElementAttribute(String name, int index, Array attval) {
        Object elem = Array.get(attval, index);
        String elemName = getNameForArrayElement(name, (new Integer(index)).toString());
        return new Attribute(elemName, elem);
    }

    /**
     * Construct a fragment who's value is contained in a Map.
     * @param name The starting name: attribute name or fragment name having as value the Map.
     * Its a fragment name, if the Map has been obtained by decomposing an encompassing value.
     * @param key the key of the value
     * @param attval the map containing the value
     * @return the created fragment
     */
    private static Attribute getMapElementAttribute(String name, Object key, Map<?, ?> attval) {
        Object item = attval.get(key);
        String itemName = getNameForItem(name, key.toString());
        return new Attribute(itemName, item);
    }

    /**
     * Construct a fragment corresponding to an item of a CompositeData.
     * @param name The starting name: attribute name or fragment name having as value the CompositeData.
     * Its a fragment name, if the CompositeData has been obtained by decomposing an encompassing value.
     * @param key the item's key
     * @param compValue the CompositeData containing the item
     * @return the created fragment
     */
    private static Attribute getCompositeItemAttribute(String name, String key, CompositeData compValue) {
        Object item = compValue.get(key);
        String itemName = getNameForItem(name, key);
        return new Attribute(itemName, item);
    }

    /**
     * Construct a fragment corresponding to a CompositeData which represents a row in a TabularData.
     * @param name The starting name: attribute name or fragment name having as value the TabularData.
     * Its a fragment name, if the TabularData has been obtained by decomposing an encompassing value.
     * @param indexes the indexes used to insert the CompositeData in the TabularData
     * @param row the CompositeData
     * @return the created fragment
     */
    private static Attribute getTabularDataAttribute(String name, Object[] indexes, CompositeData row) {
        String rowName = getNameForTabularElement(name, indexes);
        return new Attribute(rowName, row);
    }

    // =======================
    //
    // Naming policy
    //
    // =======================
    public static String DOT = ".";
    public static String COMMA = ",";
    public static String BRACKLEFT = "[";
    public static String BRACKRIGHT= "]";

    /**
     *
     * @param name attribute or fragment name
     * @return true if the name has a fragment name pattern
     */
    public static boolean isFragmentName(String name) {
        if (name.contains(DOT)) {
            return true;
        } else if (name.contains(BRACKLEFT) && name.contains(BRACKLEFT) && name.indexOf(BRACKLEFT) < name.indexOf(BRACKRIGHT)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Generate the name of a fragment corresponding to a CompositeData's item.
     * @param name composite's name
     * @param key the item's name
     */
     public static String getNameForItem(String name, String key) {
        String itemName = name + DOT + key;
        return itemName;
     }

     /**
      * Generate the name of an element of an Array.
      * @param name the name of the Array attribute or fragment
      * @param index the element's index
      * @return
      */
     public static String getNameForArrayElement(String name, String index) {
         String elemName = name + BRACKLEFT + index + BRACKRIGHT;
         return elemName;
     }

     /**
      * Generate the name of a fragment corresponding to a row of in a TabularData
      * @param name the name of the TabularData attribute or fragment
      * @param indexes the row's indexes
      * @return
      */
     public static String getNameForTabularElement(String name, Object[] indexes) {
         String s_indexes = indexes[0].toString();
         if (indexes.length > 1) {
             for (int i = 1; i < indexes.length; i++) {
                 s_indexes = s_indexes + COMMA + indexes[i];
             }
         }
         String elemName = name + BRACKLEFT + s_indexes + BRACKRIGHT;
         return elemName;
     }

     /**
      * Get the name of the attribute from a fragment's name
      * @param name fragment name
      * @return attribute name
      * @throws FragmentNameException An attribute name could not be identified as no
      * DOT nor BRACKLEFT found.
      */
     public static String getAttributeName(String name) throws FragmentNameException {
         int index; // index of the first DOT or BRACKLEFT
         int dotIndex = name.indexOf(DOT);
         int brackLeftIndex = name.indexOf(BRACKLEFT);
         if (dotIndex > 0) {
             // Got a DOT
             if (brackLeftIndex > 0) {
                 // Got both DOT and BRACKLEFT
                 if (dotIndex < brackLeftIndex) {
                     index = dotIndex;
                 } else {
                     // Check that BRACKRIGHT exist and its after the BRACKLEFT
                     if (name.indexOf(BRACKRIGHT) > 0 && name.indexOf(BRACKRIGHT) > brackLeftIndex) {
                         index = brackLeftIndex;
                     } else {
                         // BRAKs are not corresponding to indexes ... wrong format
                         throw new FragmentNameException("Can't get attribute name from: " + name);
                     }
                 }
             } else {
                 // Only got a DOT
                 index = dotIndex;
             }
         } else if (brackLeftIndex > 0) {
             // Only got BRACKLEFT
             // Check that BRACKRIGHT exist
             if (name.indexOf(BRACKRIGHT) > 0 && name.indexOf(BRACKRIGHT) > brackLeftIndex) {
                 index = brackLeftIndex;
             } else {
                 // BRAKs are not corresponding to indexes ... wrong format
                 throw new FragmentNameException("Can't get attribute name from: " + name);
             }
         } else {
             // No DOT nor BRACKLEFT
             throw new FragmentNameException("Can't get attribute name from: " + name);
         }
         return name.substring(0, index);
     }

     /**
      * Get the name of an item's key from a fragment name.
      * Suppose that the fragment name has the following format: nameDOTkey
      * @param fragmentName fragment name
      * @return key
      * @throws FragmentNameException A key name could not be identified.
      */
     public static String getItemName(String fragmentName) throws FragmentNameException {
         int dotIndex = fragmentName.indexOf(DOT);
         if (dotIndex > 0) {
             return fragmentName.substring(dotIndex + 1);
         } else {
             throw new FragmentNameException("Can't get item name from: " + fragmentName);
         }
     }

     /**
      * Get the name of an element from a fragment name.
      * Suppose that the fragment name has the following format: nameBRACKLEFTindexesBRACKRIGHT
      * @param fragmentName fragment name
      * @return the element name
      */
     public static String getElementName(String fragmentName) throws FragmentNameException {
         int brackLeftIndex = fragmentName.indexOf(BRACKLEFT);
         int brackRightIndex = fragmentName.indexOf(BRACKRIGHT);
         if (brackLeftIndex > 0 && brackRightIndex > 0 && brackLeftIndex < brackRightIndex) {
             return fragmentName;
         } else {
             throw new FragmentNameException("Can't get element name from: " + fragmentName);
         }
     }

}
