/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxCollectorService.java 8752 2011-08-23 13:41:59Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import org.apache.felix.ipojo.annotations.*;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.jmx.JmxBrowserService;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * This Collector Service creates Jmx Collectors.
 * @author durieuxp
 */
@Component(name="JmxCollectorService")
@Provides
public class JmxCollectorService implements JasmineCollectorService, JmxBrowserService {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * List of Collector
     */
    private HashMap<String, JmxCollector> collectorList = new HashMap<String, JmxCollector>();

    /**
     * List of Workers
     */
    private HashMap<String, JmxWorker> workerList = new HashMap<String, JmxWorker>();

    /**
     * List of specific Indicator Properties
     */
    private Collection<JasminePropertyInfo> properties;

    /**
     * Property names
     */
    public final static String PROP_TARGET = "target";
    public final static String PROP_MBEAN = "mbean";
    public final static String PROP_ATTR = "attr";

    /**
     * The value of the PROP_ATTR property meaning that all the
     * attributes are to be fetched.
     */
    public final static String VALUE_ALL = "all";

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="jmx")
    private String indicatorType;

    @Requires
    private JmxConnectionService jmxConnectionService = null;

    @Validate
    public void start() {
        // init the list of indicator properties (needed for getPropertiesInfo)
        // these properties will be passed with the JasmineIndicator at getCollector()
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_TARGET, "Jmx target name", true));
        properties.add(new JasminePropertyInfo(PROP_MBEAN, "MBean Jmx filter", true));
        properties.add(new JasminePropertyInfo(PROP_ATTR, "Comma separated list of attributes", false));
    }

    @Invalidate
    public void stop() {
        // TODO
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for this JasmineIndicator
     * @param indicator
     * @param period in second
     * @param probeid
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid) throws JasmineCollectorException {
        // Build a key to identify the Collector
        // The Collector will be dedicated to the probe that use it.
        String key = indicator.getName() + probeid;

        // Create the Collector if not existing yet
        JmxCollector collector = collectorList.get(key);
        if (collector == null) {
            // sanity check, not really needed.
            if (! indicator.getType().equals(indicatorType)) {
                logger.error("Bad type: " + indicator.getType());
                throw new JasmineCollectorException("Bad type: " + indicator.getType());
            }

            // Get Jmx specifics parameters
            Map<String, String> props = indicator.getProperties();
            String target = props.get(PROP_TARGET);
            String mbean = props.get(PROP_MBEAN);
            String attr = props.get(PROP_ATTR);

            // Get the JasmineTarget
            JasmineTarget jt = probeManager.getTarget(target);
            if (jt == null) {
                logger.error("Unknown target: " + target);
                throw new JasmineCollectorException("Unknown target: " + target);
            }

            // Create the JmxCollector
            collector = new JmxCollector(key, indicator, period, mbean, attr);
            collectorList.put(key, collector);

            // Find the JmxWorker to be used :
            // In this implementation, we have one worker for each target.
            JmxWorker worker = workerList.get(target);
            if (worker == null) {
                worker = new JmxWorker(jt, jmxConnectionService);
                workerList.put(target, worker);
            }

            // Notify the JmxWorker that a new JmxCollector exists
            worker.addCollector(collector);
        }
        return collector;
    }

    /**
     * Retrieve the description of all specific properties hidden in JasmineCollector
     * Return a Collection of all properties with their description
     * @return  a Collection of property descriptions
     */
    public Collection<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }
}
