/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxUtil.java 8816 2011-09-08 09:30:10Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.JMException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.openmbean.CompositeData;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Support for access to MBean attributes and attribute fragments.
 * @author danesa
 */
public class JmxUtil {

    private static Log logger = LogFactory.getLog(JmxCollector.class);

    /**
     * Get all the MBeans corresponding to an ObjectName pattern.
     * When the pattern is null or no domain and key properties are specified,
     * all objects are selected (and filtered if a query is specified).
     * It returns the set of ObjectNames for the MBeans selected.
     * @param cnx Connection to the MBean server
     * @param on ObjectName pattern
     * @return the set of ObjectNames corresponding to the pattern
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     */
    public static Set<ObjectName> getMBeans(final MBeanServerConnection cnx, ObjectName on) throws IOException {
        QueryExp query = null;
        return cnx.queryNames(on, query);
    }

    /**
     * Return a list of attribute names of a given MBean.
     *
     * @param cnx MBeanServer connection
     * @param on The MBean's name.
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException A JMX exception occurred when using the MBean.
     */
    public static List<String> getMBeanAttNames(final MBeanServerConnection cnx, ObjectName on) throws IOException, JMException {
        List<String> attNames = new ArrayList<String>();

        MBeanInfo mbeanInfo = cnx.getMBeanInfo(on);
        MBeanAttributeInfo[] attsInfo = mbeanInfo.getAttributes();
        for (MBeanAttributeInfo attInfo : attsInfo) {
            String attName = attInfo.getName();
            attNames.add(attName);
        }
        return attNames;
    }

    /**
     * Return an Attribute object corresponding to the given attribute fragment name.
     * @param cnx Connection to the MBean server
     * @param objname MBean to poll
     * @param name fragment name
     * @return the Attribute object containing the fragment's value. The returned
     * Attribute object's name, is the given name.
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException Problem with the JMX code.
     * @throws FragmentNameException exception related to the fragment naming policy
     */
    public static Attribute getAttributeFragment(MBeanServerConnection cnx, ObjectName objname, String name) throws IOException, JMException, FragmentNameException {
        String attName = FragmentUtil.getAttributeName(name);
        logger.debug("Fragment name: {0}, Attribute name: {1}, MBean OBJECT_NAME: {2}" , name, attName, objname);
        try {
            // suppose the fragment is an item element
            // i.e. the name has the following format:
            // attName.itemName
            String itemName = FragmentUtil.getItemName(name);
            logger.debug("Gen Attribute for item {0}", itemName);
            return genItemAttribute(cnx, objname, attName, itemName);
        } catch (FragmentNameException fe) {
            try {
                // suppose the fragment is an indexed element
                // i.e. the name has the following format:
                // attName[index]
                String elemName = FragmentUtil.getElementName(name);
                logger.debug("Gen Attribute for element {0}", elemName);
                return genElemAttribute(cnx, objname, attName, elemName);
            } catch (FragmentNameException fee) {
                throw new FragmentNameException(fe + " - " + fee);
            }
        }
    }

    /**
     * Construct an Attribute corresponding to an attribute's item.
     * @param cnx Connection to the MBean server
     * @param objname MBean to poll
     * @param attName the attribute's name
     * @param key the item's key
     * @return the created Attribute instance
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException Problem with the JMX code.
     * @throws FragmentNameException exception related to the fragment naming policy
     */
    public static Attribute genItemAttribute(MBeanServerConnection cnx, ObjectName objname, String attName, String key) throws IOException, JMException, FragmentNameException {
        if (FragmentUtil.isFragmentName(key)) {
            // TODO add support for multi-level fragment names
            // For example a.b.c or a.b[2]
            logger.warn("Support for composite key {0} not implemented yet !", key);
            return null;
        } else {
            Object attValue = cnx.getAttribute(objname, attName);
            if (attValue instanceof CompositeData) {
                CompositeData compValue = (CompositeData) attValue;
                Set<String> keys = compValue.getCompositeType().keySet();
                for (String _key : keys) {
                    if (key.equals(_key)) {
                        Object item = compValue.get(key);
                        String itemAttName = FragmentUtil.getNameForItem(attName, key);
                        return new Attribute(itemAttName, item);
                    }
                }
                throw new FragmentNameException("Attribute " + attName + " of MBean " + objname + " has a CompositeData value, but non of its keys is equal to " + key);
            } else if (attValue instanceof Map) {
                Map<?, ?> mapValue = (Map<?, ?>) attValue;
                Set<?> keys = mapValue.keySet();
                for (Object _key : keys) {
                    if (key.equals(_key)) {
                        Object item = mapValue.get(key);
                        String itemAttName = FragmentUtil.getNameForItem(attName, key);
                        return new Attribute(itemAttName, item);
                    }
                }
                throw new FragmentNameException("Attribute " + attName + " of MBean " + objname + " has a Map type value, but non of its keys is equal to " + key);
            } else {
                throw new FragmentNameException("Attribute " + attName + " of MBean " + objname + " has neither a Map type value, nor a CompositeData value !");
            }

        }
    }

    /**
     * Construct an Attribute corresponding to an indexed attribute's element.
     * @param cnx Connection to the MBean server
     * @param objname MBean to poll
     * @param attName the attribute's name
     * @param elemName indexed elemen's name (expected format is name[indexes])
     * @param key the item's key
     * @return the created Attribute instance
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException Problem with the JMX code.
     * @throws FragmentNameException exception related to the fragment naming policy
     */
    public static Attribute genElemAttribute(MBeanServerConnection cnx, ObjectName objname, String attName, String elemName) throws IOException, JMException, FragmentNameException {
        Object attValue = cnx.getAttribute(objname, attName);
        if (!attValue.getClass().isArray()) {
            throw new FragmentNameException("Attribute " + attName + " in MBean " + objname + " is not array, so cannot get element " + elemName);
        }
        int brackLeftIndex = elemName.indexOf(FragmentUtil.BRACKLEFT);
        int brackRightIndex = elemName.indexOf(FragmentUtil.BRACKRIGHT);
        String indexes = elemName.substring(brackLeftIndex + 1, brackRightIndex);
        logger.debug("Indexes: ", indexes.toString());
        if (indexes.contains(FragmentUtil.COMMA)) {
            // TODO support multi indexed, bu also a thing such as a[i1,i2].b.c ..
            logger.warn("Support for multi-indexed elements {0} not implemented yet !", elemName);
            return null;
        } else {
            if (brackRightIndex == elemName.length() -1) {
                try {
                    int index = Integer.parseInt(indexes);
                    try {
                        Object[] sArray = (Object[]) attValue;
                        return new Attribute(elemName, sArray[index]);
                    } catch (ClassCastException cc) {
                        // try generic Array
                        try {
                            Array arrayval = (Array) attValue;
                            Object elem = Array.get(arrayval, index);
                            return new Attribute(elemName, elem);
                        } catch (ClassCastException ccc) {
                            logger.warn("Support for indexed elements other then Array or Object[] not implemented yet ! Unsupported element: ", elemName);
                            return null;
                        }
                    }
                } catch (NumberFormatException nfe) {
                    throw new FragmentNameException("Element " + elemName + " of attribute " + attName + " in MBean " + objname + " has unsupported index form.");
                }
            } else {
                logger.warn("Support for complex elements {0} not implemented yet !", elemName);
                return null;
            }

        }
    }
}
