/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxWorker.java 8851 2011-09-13 08:14:48Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.JMException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionFactory;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.jasmine.probe.jmxconnection.simple.JmxConnectionServiceImpl;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Poll a Jmx target to get mbean attributes values.
 * @author durieuxp
 */
public class JmxWorker extends Thread {
    private Log logger = LogFactory.getLog(JCollector.class);

    private boolean started = false;

    /**
     * The name of the monitored target.
     */
    private String targetName;

    /**
     * The target's JMX URL.
     */
    private String url;

    /**
     * Period used to check for values to get.
     * TODO: Could be computed from period values
     */
    private long collectorTimeout = DEFAULT_TIMEOUT;
    private static final long DEFAULT_TIMEOUT = 1000; // 1 sec

    /**
     * List of Collectors managed by this Worker
     */
    private Collection<JmxCollector> collectors = new ArrayList<JmxCollector>();

    private JmxConnectionService jmxService;

    private Map<String, String> props = new HashMap<String, String>();

    private JmxConnectionFactory cfactory = null;

    /**
     * The name of the managed server represented by the target.
     */
    private String serverName = null;

    /**
     * The name of the management domain to which belongs the target server.
     */
    private String domainName = null;

    /**
     * Constructor
     */
    public JmxWorker(JasmineTarget target, JmxConnectionService jmxConnectionService) {
        logger.debug(target.getName());
        this.targetName = target.getName();
        this.url = target.getUrl();
        this.props = target.getProperties();
        this.jmxService = jmxConnectionService;
    }

    /**
     * remove it
     */
    public synchronized void remove() {
        started = false;
    }

    /**
     * Add a new Collector to this worker.
     */
    public synchronized void addCollector(JmxCollector col) {
        // Add the Collector to the list
        collectors.add(col);

        // Starts the Thread if not started yet.
        if (! started) {
            started = true;
            start();
        }
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (started) {
            // Check if must fetch some new results
            Collection<JmxCollector> topoll = new ArrayList<JmxCollector>();
            synchronized(this) {
                for (JmxCollector coll : collectors) {
                    if (coll.needResult()) {
                        topoll.add(coll);
                    }
                }
            }

            if (! topoll.isEmpty()) {
                // Keep timestamp identical for all these polling
                long t = System.currentTimeMillis();

                // Get a MBeanServerConnection
                MBeanServerConnection cnx = null;
                try {
                    cnx = getMBeanServerConnection();
                } catch (JmxConnectionException e) {
                    logger.warn("Cannot get Jmx Access:" + e);
                }
                if (cnx != null) {
                    if ((serverName == null) || (domainName == null)) {
                        setServerAndDomain(cnx);
                    }
                    // Fetch the results (Metrics) (outside the lock)
                    // this may block in case the target is off.
                    for (JmxCollector coll : topoll) {
                        List<Metric> metrics = collect(cnx, coll);
                        coll.addResult(metrics);
                    }
                }
                // release the MBeanServerConnection
                // nothing to do today...
            }

            // Wait
            synchronized(this) {
                try {
                    wait(collectorTimeout);
                } catch (InterruptedException e) {
                    logger.warn(getName() + ": collector interrupted", e);
                } catch (Exception e) {
                    logger.warn(getName() + ": collector exception", e);
                }
            }
        }
    }

    // -----------------------------------------------------------------------------------------
    // Private methods
    // -----------------------------------------------------------------------------------------

    /**
     * Get a MBeanServerConnection
     * @return MBeanServerConnection
     */
    private MBeanServerConnection getMBeanServerConnection() throws JmxConnectionException {
        if (cfactory == null) {
            cfactory = jmxService.getJmxConnectionFactory(url, props);
        }
        return cfactory.getMBeanServerConnection();
    }

    private void setServerAndDomain(MBeanServerConnection cnx) {
        try {
            ObjectName j2eeinstance = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
            Iterator<?> onames = cnx.queryNames(j2eeinstance, null).iterator();
            ObjectName serverOn = (ObjectName) onames.next();
            this.serverName = (String) cnx.getAttribute(serverOn, "serverName");
            this.domainName = (String) serverOn.getDomain();
        } catch (Exception e) {
            logger.warn("Cannot get server and domain names for target ", targetName);
        }

    }

    /**
     * Collect values of MBean attributes for this Collector.
     * @param cnx MBeanServerConnection
     * @param col JmxCollector
     * @return the Metrics for (all) the MBean(s) corresponding to the collector's
     * ObjectName pattern. All the MBeans correspond to one target ; so all the
     * Metrics have the same target.
     */
    private List<Metric> collect(MBeanServerConnection cnx, JmxCollector col) {
        LinkedList<Metric> metrics = new LinkedList<Metric>();
        try {
            ObjectName objname = col.getObjectName();
            if (objname.isPattern()) {
                logger.debug("pattern:" + objname.toString());
                // Must loop on all the mbeans matching the pattern
                Set<ObjectName> onset = JmxUtil.getMBeans(cnx, objname);
                for (ObjectName mbean : onset) {
                    Metric m = getMBeanMetric(cnx, mbean, col);
                    if (m != null) {
                        metrics.add(m);
                    }
                }
            } else {
                logger.debug("single mbean:" + objname.toString());
                Metric m = getMBeanMetric(cnx, objname, col);
                if (m != null) {
                    metrics.add(m);
                }
            }
        } catch (Exception e) {
            // TODO
            logger.warn("getAttributes error on JmxServer: " + e);
        }
        logger.debug("nb of metrics found: " + metrics.size());
        return metrics;
    }

    /**
     * Poll one MBean's attributes and get the values of the attributes and/or the fragments
     * corresponding to the collector's <code>attrlist</code>. If the indicator definition
     * does not provides an attributes/fragments list, then all the possible attributes are polled.
     * @param cnx Connection to the MBean server
     * @param objname The MBean's ObjectName
     * @param col The current collector
     * @return A Metric object containing the attributes' and/or fragments' values.
     * @throws IOException
     * @throws JMException
     */
    private Metric getMBeanMetric(MBeanServerConnection cnx, ObjectName objname, JmxCollector col) throws IOException, JMException {
        logger.debug("mbean:" + objname.toString());
        if (!cnx.isRegistered(objname)) {
            logger.error("Cannot get attributes in MBean {0}", objname);
        }
        Metric metric = null;
        AttributeList attl = null;
        long time = System.currentTimeMillis();
        if (col.hasAttributes()) {
            /**
             * For each provided attribute name and/or attribute fragment name, get its value
             * and construct with it a JMX Attribute instance. Put these instances into
             * the <code>attl</code> AttributeList.
             */
            attl = new AttributeList();
            for (String name : col.geAttrlist()) {
                Attribute att = null;
                if (FragmentUtil.isFragmentName(name)) {
                    try {
                        att = JmxUtil.getAttributeFragment(cnx, objname, name);
                    } catch (FragmentNameException e) {
                        logger.error("Cannot get value for {0} in MBean {1}", name, objname);
                    }
                } else {
                    // <code>name</code> is an attribute's name
                    try {
                        AttributeList atts =  cnx.getAttributes(objname, new String[] { name });
                        if (atts.isEmpty()) {
                            logger.error("Cannot get value for {0} in MBean {1} (there is no attribute having this name)");
                        } else {
                            att = (Attribute) atts.get(0);
                        }
                    } catch (InstanceNotFoundException ae) {
                        logger.error("Cannot get value for {0} in MBean {1} (there is no such MBean)");
                    }
                }
                if (att != null) {
                    attl.add(att);
                }
            }
        } else {
            // if no attributes were provided, get the possible attribute names
            // TODO Exception support
            List<String> attNameList = JmxUtil.getMBeanAttNames(cnx, objname);
            String[] attNames = new String[attNameList.size()];
            int i = 0;
            for (String attName : attNameList) {
                attNames[i++] = attName;
            }
            attl = cnx.getAttributes(objname, attNames);
        }
        if (attl.isEmpty()) {
            logger.warn("No metrics found for indicator: " + col.getIndicator().getName()  + ", for mbean: " + objname.toString() + ", and target: " + targetName);
        } else {
            metric = new Metric(time, objname, attl, targetName);
            // add metadata
            metric.setProperty(JmxConnectionServiceImpl.PROP_SERVER, serverName);
            metric.setProperty(JmxConnectionServiceImpl.PROP_DOMAIN, domainName);
            metric.setProperty(JmxConnectionServiceImpl.PROP_URL, url);
        }
        return metric;
    }

}
