/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxCollector.java 8850 2011-09-13 08:12:22Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.management.Attribute;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleBooleanResult;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.JasmineSingleStringResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.simple.JmxConnectionServiceImpl;
import org.ow2.jasmine.probe.util.CsvConverter;

/**
 * Collector implementation for JMX.
 * This implementation works with a separate thread used to pre-collect the results.
 * Using a thread is interesting when we do not want to be stuck in case info
 * cannot be got because the target is off.
 * @author durieuxp
 */
public class JmxCollector extends JCollector {

    /**
     * Jmx MBean Filter (ObjectName syntax)
     */
    private String pattern;

    /**
     *
     */
    private ObjectName objname = null;

    /**
     * The name of the attributes or attribute fragments to get.
     */
    private Collection<String> attrlist;

    /**
     * Last result got by JmxWorker
     */
    private List<Metric> metrics = null;

    /**
     * Last value returned, kept in case a probe need it twice.
     */
    private JasmineIndicatorValue cached = null;

    /**
     * Pass stopped to true if the probe using this collector is stopped.
     */
    private boolean stopped = false;

    /**
     * Constructor
     * @param indicator definition
     */
    public JmxCollector(String name, JasmineIndicator indicator, int period, String mbean, String csv) {
        super(name, indicator, period);
        this.pattern = mbean;
        this.attrlist = CsvConverter.csv2list(csv);
    }

    /**
     * Getter for attrlist (list of provided attribute and or fragment names).
     * @return return the provided attribute and or fragment names.
     */
    public Collection<String> geAttrlist() {
        return attrlist;
    }

    /**
     * @return The ObjectName instance corresponding to the mbean pattern provided in
     * the indicator definition.
     * @throws JasmineCollectorException an ObjectName could not be obtained as the pattern
     * in not well formed given the JMX naming conventions.
     */
    public ObjectName getObjectName() throws JasmineCollectorException {
        if (objname == null) {
            try {
                objname = ObjectName.getInstance(pattern);
            } catch (MalformedObjectNameException e) {
                throw new JasmineCollectorException("Malformed pattern: " + pattern + " (" + e.toString() + " )");
            } catch (NullPointerException e) {
                throw new JasmineCollectorException("Null pattern");
            }
        }
        return objname;
    }

    /**
     * Allows the JmxWorker to know if attributes and/or fragments are provided by the indicator definition.
     * @return true if attributes and or fragments are provided by the indicator definition.
     */
    protected boolean hasAttributes() {
        if (attrlist == null || attrlist.isEmpty()) {
            return false;
        } else {
            // if one element equal to "all" in the attrlist
            if (attrlist.size() == 1) {
                for (String elem : attrlist) {
                    if (JmxCollectorService.VALUE_ALL.equals(elem)) {
                        return false;
                    }
                }
            }
            return true;
        }
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * This method return a List of results in case indicator represents
     * actually a list of values.
     * @return The constructed JasmineIndicatorValue
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        // First of all, return the cached value if any
        if (cached != null) {
            logger.debug("cached value returned");
            return cached;
        }
        if (metrics == null) {
            logger.warn("No result ready to return");
            return null;
        }
        // Create the JasmineIndicatorValue instance to return
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        String name = getIndicator().getName();
        logger.debug(name);
        jiv.setName(name);
        // fill in JasmineIndicValue instance with JasmineSingleResults corresponding to
        // the fragments composing the metrics.
        for (Metric metric : metrics) {
            // Each metric corresponds to a MBean.
            if (metric.getAttributeList().size() == 0) {
                logger.warn("Result with no value: " + metric.toString());
                continue;
            }
            for (Iterator it = metric.getAttributeList().iterator(); it.hasNext(); ) {
                Attribute att = (Attribute) it.next();
                if (att.getValue() == null) {
                    // No value: forget this attribute
                    logger.warn("No value for " + att.getName() + " in this indicator: " + getName());
                    continue;
                }

                // decompose each attribute into elementary fragments
                List<Attribute> decomposedList = FragmentUtil.getFragments(att);

                for (Attribute dAtt : decomposedList) {
                    // TODO add metadata + properties
                    JasmineSingleResult jsr = getJsr(dAtt, metric.getTimestamp(), indicator.getScale());
                    // Add MBean name as "mbean" property
                    jsr.addProperty(JmxConnectionServiceImpl.PROP_MBEAN, metric.getMBean().toString());
                    if (jsr != null) {
                        jiv.addValue(jsr);
                    }
                }
            }
        }
        /**
         * All the Metrics have the same values for the following metadata :
         * JmxConnectionServiceImpl.PROP_TARGET
         * JmxConnectionServiceImpl.PROP_SERVER - already set in Metric.properties
         * JmxConnectionServiceImpl.PROP_DOMAIN - already set in Metric.properties
         * JmxConnectionServiceImpl.PROP_URL    - already set in Metric.properties
         * Use the first Metric instance to set metadata into the jiv.
         */
        jiv.addMetadata(JmxConnectionServiceImpl.PROP_TARGET, metrics.get(0).getTarget());
        String key = JmxConnectionServiceImpl.PROP_SERVER;
        jiv.addMetadata(key, metrics.get(0).getProperties().get(key));
        key = JmxConnectionServiceImpl.PROP_DOMAIN;
        jiv.addMetadata(key, metrics.get(0).getProperties().get(key));
        key = JmxConnectionServiceImpl.PROP_URL;
        jiv.addMetadata(key, metrics.get(0).getProperties().get(key));
        cached = jiv;
        metrics = null;
        return jiv;
    }

    /**
     * Return the JasmineSingleResult corresponding to an MBean attribute.
     * Currently, a JSR is returned only for Numbers, Strings and Booleans.
     * @param att an MBea's attribute
     * @param timestamp time stamp to put into the JSR
     * @param scale scale to use when calculating the value to put into the JSR
     * @return A JSR corresponding to the attribute's value or null if the value could not be
     * calculated (in the current implementation, if the value is not a number).
     * @throws JasmineCollectorException Raised by divideValues.
     */
    private JasmineSingleResult getJsr(Attribute att, long timestamp, int scale) throws JasmineCollectorException {
        JasmineSingleResult jsr = null;
        Object value = att.getValue();
        if (value instanceof String) {
            String s = (String) value;
            jsr = new JasmineSingleStringResult();
            jsr.setValue(s);
        } else if (value instanceof Boolean) {
            Boolean b = (Boolean) value;
            jsr = new JasmineSingleBooleanResult();
            jsr.setValue(b);
        } else {
            // try a Number
            try {
                Number numberValue = (Number) value;
                jsr = new JasmineSingleNumberResult();
                if (scale != 1) {
                    numberValue = divideValues(numberValue, scale);
                }
                jsr.setValue(numberValue);
            } catch (java.lang.ClassCastException cce) {
                logger.warn("Attribute " + att.getName() + " is not a number, nor a String. Currently no result (JasmineSingleResult) returned for it !");
                return null;
            }
        }
        jsr.setName(att.getName());
        jsr.setTimestamp(timestamp);
        return jsr;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        stopped = true;
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("");
        stopped = false;
    }

    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    /**
     * Check if this Collector needs a new result.
     * A stopped collector returns false.
     * Otherwise, it returns true if there are nor metrics, or if metrics are out-dated
     * regarding to the poll period.
     *
     * @return true if it needs one
     */
    public boolean needResult() {
        if (stopped) {
            return false;
        }
        if (metrics == null) {
            return true;
        }
        long timestamp = metrics.get(0).getTimestamp();
        if (oldvalue(timestamp)) {
            // TODO ? to cheque with Ph.
            // forget this value too old.
            // metrics = null;
            return true;
        }
        return false;
    }

    /**
     * Add a new result to the list.
     * @param metrics
     */
    public void addResult(List<Metric> metrics) {
        if (metrics.size() == 0) {
            logger.warn("Empty metric list");
            return;
        }
        logger.debug("");
        this.metrics = metrics;
        this.cached = null;
        // TODO maybe better to compute cached now, instead of in getLastValue ?
    }

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * @return true if this timestamp is too old for the period.
     */
    private boolean oldvalue(long timestamp) {
        long now = System.currentTimeMillis();
        return (now - timestamp > period * 1000);
    }


}
