/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxUtil.java 8989 2011-09-26 07:10:14Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.JMException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.TabularData;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Support for access to MBean attributes and attribute fragments.
 * @author danesa
 */
public class JmxUtil {

    private static Log logger = LogFactory.getLog(JmxCollector.class);

    /**
     * Get all the MBeans corresponding to an ObjectName pattern.
     * When the pattern is null or no domain and key properties are specified,
     * all objects are selected (and filtered if a query is specified).
     * It returns the set of ObjectNames for the MBeans selected.
     * @param cnx Connection to the MBean server
     * @param on ObjectName pattern
     * @return the set of ObjectNames corresponding to the pattern
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     */
    public static Set<ObjectName> getMBeans(final MBeanServerConnection cnx, ObjectName on) throws IOException {
        QueryExp query = null;
        return cnx.queryNames(on, query);
    }

    /**
     * Return a list of attribute names of a given MBean.
     *
     * @param cnx MBeanServer connection
     * @param on The MBean's name.
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException A JMX exception occurred when using the MBean.
     */
    public static List<String> getMBeanAttNames(final MBeanServerConnection cnx, ObjectName on) throws IOException, JMException {
        List<String> attNames = new ArrayList<String>();

        MBeanInfo mbeanInfo = cnx.getMBeanInfo(on);
        MBeanAttributeInfo[] attsInfo = mbeanInfo.getAttributes();
        for (MBeanAttributeInfo attInfo : attsInfo) {
            String attName = attInfo.getName();
            attNames.add(attName);
        }
        return attNames;
    }

    /**
     * Create a new Attribute instance for an indexed element. The attribute's value is the value of the element having the given index.
     * The atrtibute's name is generated from the container's name (the name associated to the indexed object) and
     * the index value.
     * @param contName the name associated to the indexed object
     * @param index the element's index
     * @param elemValue the value of the element having the given index
     * @return the created Attribute instance
     */
    public static Attribute createNewIndexedAttribute(String contName, String index, Object elemValue) {
        String attName = FragmentUtil.nameForIndexElement(contName, index);
        Object attValue = elemValue;
        return new Attribute(attName, attValue);
    }

    /**
     * Create a new Attribute instance for a keyed item. The attribute's value is the value of the item having the given key.
     * The atrtibute's name is generated from the container's name and the key value.
     * @param contName the name associated to the composite object (the container)
     * @param key the item's key
     * @param itemValue the value of the item having the given key
     * @return the created Attribute instance
     */
    public static Attribute createNewItemAttribute(String contName, String key, Object itemValue) {
        String attName = FragmentUtil.nameForItem(contName, key);
        Object attValue = itemValue;
        return new Attribute(attName, attValue);
    }

    /**
     * Return an Attribute object corresponding to the given attribute fragment name.
     * @param cnx Connection to the MBean server
     * @param objname MBean to poll
     * @param name attribute fragment name
     * @param rootAtt the MBean attribute containing this fragment
     * @return the Attribute object containing the fragment's value. The returned
     * Attribute object's name, is the given name.
     * @throws IOException A communication problem occurred when
     * talking to the MBean server.
     * @throws JMException Problem with the JMX code.
     * @throws FragmentNameException exception related to the fragment naming policy
     */
    public static Attribute getAttributeFragment(String name, Attribute att) throws FragmentNameException {
        if (name.equals(att.getName())) {
            return att;
        }
        if (FragmentUtil.nextElementIsKey(name, att.getName())) {
            String key = FragmentUtil.getKeyName(name, att.getName());
            try {
                Object itemValue = ComplexValuesUtil.getItem(att.getValue(), key);
                Attribute newAtt = createNewItemAttribute(att.getName(), key, itemValue);
                return getAttributeFragment(name, newAtt);
            } catch (ComplexValuesException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
                throw new FragmentNameException("Could not extract an attribute fragment having name: " + name + ". The source attribute has name: " + att.getName());
            }
        }

        if (FragmentUtil.nextElementIsIndex(name, att.getName())) {
            String indexes = FragmentUtil.getIndexes(name, att.getName());
            try {
                Object indexedValue = ComplexValuesUtil.getElement(att.getValue(), indexes);
                Attribute newAtt = createNewIndexedAttribute(att.getName(), indexes, indexedValue);
                return getAttributeFragment(name, newAtt);
            } catch (ComplexValuesException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
                throw new FragmentNameException("Could not extract an attribute fragment having name: " + name + ". The source attribute has name: " + att.getName());
            }
        }

        // wrong format
        throw new FragmentNameException("Could not extract an attribute fragment having name: " + name + ". The source attribute has name: " + att.getName());
    }

    /**
     * Decompose a JMX Attribute having a complex type into a list of Attributes having
     * simple types (the fragments).
     * The supported complex types are:
     * <ul>
     *   <li>javax.management.openmbean.CompositeData</li>
     *   <li>javax.management.openmbean.TabularData</li>
     *   <li>Array</li>
     *   <li>String[]</li>
     *   <li>Map</li>
     *   <li>List</li>
     * </ul>
     * The supported simple types are:
     * <ul>
     *   <li>Number</li>
     *   <li>Boolean</li>
     *   <li>String</li>
     * </ul>
     *
     * @param att The initial attribute which may have a complex or a simple type.
     * It its value has a simple type, the attribute is returned as it is.
     * If it has a complex type, its value is decomposed in simple elements, the fragments.
     *
     * @return A list of fragments. A fragment is contained in an Attribute instance.
     * A fragment's value is a simple value obtained by a complex value decomposition.
     * A fragment's name is a name is based on the fragments naming policy.
     * It starts with the initial attribute's name and contains a representation of the path
     * followed by the decomposition process.
     */
    public static List<Attribute> getAttributeFragments(Attribute att) throws ComplexValuesException  {
        List<Attribute> result = new ArrayList<Attribute>();
        String name = att.getName();
        Object value = att.getValue();
        if (ComplexValuesUtil.hasSimpleType(value)) {
            logger.debug("Return final fragment with name: " + name + " and value: " + value);
            result.add(att);
            return result;
        }
        if (value instanceof CompositeData) {
            CompositeData compValue = (CompositeData) value;
            Set<String> keys = compValue.getCompositeType().keySet();
            for (String key : keys) {
                Object item = ComplexValuesUtil.getCompositeItem(compValue, key);
                String itemName = FragmentUtil.getNameForItem(name, key);
                Attribute newAttribute = new Attribute(itemName, item);
                result.addAll(getAttributeFragments(newAttribute));
            }
            return result;
        }
        if (value instanceof TabularData) {
            TabularData tabData = (TabularData) value;
            Collection<?> values = tabData.values();
            for (Object tabValue : values) {
                // the rows in the TabularData are CompositeData
                CompositeData compValue = (CompositeData) tabValue;
                Object[] indexes = tabData.calculateIndex(compValue);
                String elemName = FragmentUtil.getNameForTabularElement(name, indexes);
                Attribute newAttribute = new Attribute(elemName, compValue);
                result.addAll(getAttributeFragments(newAttribute));
            }
            return result;
        }
        if (value instanceof Map) {
            Map<?, ?> mapVal = (Map<?, ?>) value;
            Set<?> keys = mapVal.keySet();
            for (Object key : keys) {
                String skey = key.toString();
                Object item = ComplexValuesUtil.getMapItem(mapVal, skey);
                String itemName = FragmentUtil.getNameForItem(name, skey);
                Attribute newAttribute = new Attribute(itemName, item);
                result.addAll(getAttributeFragments(newAttribute));
            }
            return result;
        }
        if (value instanceof List) {
            List listVal = (List) value;
            for (int i = 0; i < listVal.size(); i++) {
                String index = (new Integer(i)).toString();
                Object item = ComplexValuesUtil.getListElement(listVal, index);
                String itemName = FragmentUtil.getNameForItem(name, index);
                Attribute newAttribute = new Attribute(itemName, item);
                result.addAll(getAttributeFragments(newAttribute));
            }
            return result;
        }
        if (value.getClass().isArray()) {
            try {
                Object[] oArray = (Object[]) value;
                for (int i = 0; i < oArray.length; i++) {
                    Object elem = oArray[i];
                    String elemName = FragmentUtil.getNameForArrayElement(name, (new Integer(i)).toString());
                    Attribute newAttribute = new Attribute(elemName, elem);
                    result.addAll(getAttributeFragments(newAttribute));
                }
                return result;
            } catch (ClassCastException cc) {
                // try generic Array
                Array arrayval = (Array) value;
                for (int i = 0; i < Array.getLength(arrayval); i++) {
                    Object elem = Array.get(arrayval, i);
                    String elemName = FragmentUtil.getNameForArrayElement(name, (new Integer(i)).toString());
                    Attribute newAttribute = new Attribute(elemName, elem);
                    result.addAll(getAttributeFragments(newAttribute));
                }
                return result;
            }
        }
        throw new ComplexValuesException("Complex elements other then CompositeData, TabularData, Map, List, Array or Object[] are not supported");
    }
}
