/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxWorker.java 9042 2011-10-03 12:23:48Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.JMException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionFactory;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Poll a Jmx target to get mbean attributes values.
 * @author durieuxp
 */
public class JmxWorker extends Thread {
    private Log logger = LogFactory.getLog(JCollector.class);

    private boolean started = false;

    /**
     * The name of the monitored target.
     */
    private String targetName;

    /**
     * The target's JMX URL.
     */
    private String url;

    /**
     * Period used to check for values to get.
     * TODO: Could be computed from period values
     */
    private long collectorTimeout = DEFAULT_TIMEOUT;
    private static final long DEFAULT_TIMEOUT = 1000; // 1 sec

    /**
     * List of Collectors managed by this Worker
     */
    private Collection<JmxCollector> collectors = new ArrayList<JmxCollector>();

    private JmxConnectionService jmxService;

    private Map<String, String> props = new HashMap<String, String>();

    private JmxConnectionFactory cfactory = null;

    /**
     * The name of the managed server represented by the target.
     */
    private String serverName = null;

    /**
     * The name of the management domain to which belongs the target server.
     */
    private String domainName = null;


    /**
     * Pass to true if arrived to set server and domain names.
     * Avoids useless iterations.
     */
    private boolean serverNameSet = false;

    /**
     * Keeps last error message when trying to set server and domain name.
     * Avoids useless warnings.
     */
    private String error;

    /**
     * Constructor a worker for a given target.
     */
    public JmxWorker(JasmineTarget target, JmxConnectionService jmxConnectionService) {
        logger.debug(target.getName());
        this.targetName = target.getName();
        this.url = target.getUrl();
        this.props = target.getProperties();
        this.jmxService = jmxConnectionService;
    }

    /**
     * remove it
     */
    public synchronized void remove() {
        started = false;
    }

    /**
     * Add a new Collector to this worker.
     */
    public synchronized void addCollector(JmxCollector col) {
        // Add the Collector to the list
        collectors.add(col);

        // Starts the Thread if not started yet.
        if (! started) {
            started = true;
            start();
        }
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (started) {
            // Check the collectors which need results to be checked
            Collection<JmxCollector> topoll = new ArrayList<JmxCollector>();
            synchronized(this) {
                for (JmxCollector coll : collectors) {
                    if (coll.needResult()) {
                        topoll.add(coll);
                    }
                }
            }

            if (!topoll.isEmpty()) {
                // Keep timestamp identical for all these polling
                long t = System.currentTimeMillis();

                // Get a MBeanServerConnection
                MBeanServerConnection cnx = null;
                try {
                    cnx = getMBeanServerConnection();
                } catch (JmxConnectionException e) {
                    logger.warn("Cannot get Jmx Access:" + e);
                }
                if (cnx != null) {
                    setServerAndDomain(cnx);

                    // Fetch the results (Metrics) (outside the lock)
                    // this may block in case the target is off.
                    for (JmxCollector coll : topoll) {
                        List<Metric> metrics = collect(cnx, coll);
                        coll.addResult(metrics);
                    }
                }
                // release the MBeanServerConnection
                // nothing to do today...
            }

            // Wait
            synchronized(this) {
                try {
                    wait(collectorTimeout);
                } catch (InterruptedException e) {
                    logger.warn(getName() + ": collector interrupted", e);
                } catch (Exception e) {
                    logger.warn(getName() + ": collector exception", e);
                }
            }
        }
    }

    // -----------------------------------------------------------------------------------------
    // Private methods
    // -----------------------------------------------------------------------------------------

    /**
     * Get a MBeanServerConnection
     * @return MBeanServerConnection
     */
    private MBeanServerConnection getMBeanServerConnection() throws JmxConnectionException {
        if (cfactory == null) {
            cfactory = jmxService.getJmxConnectionFactory(url, props);
        }
        return cfactory.getMBeanServerConnection();
    }

    private void setServerAndDomain(MBeanServerConnection cnx) {
        if (serverNameSet) {
            return;
        }
        try {
            this.domainName = JmxCollectorService.DEFAULT_DOMAIN;
            this.serverName = JmxCollectorService.DEFAULT_SERVER;
            /**
             * Consider particular target cases:
             */
            /**
             * Java EE Server
             * --------------
             */
            ObjectName on = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
            Set ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                Iterator<?> onames = ons.iterator();
                ObjectName serverOn = (ObjectName) onames.next();
                this.domainName = (String) serverOn.getDomain();
                String name = serverOn.getKeyProperty("name");
                if (name != null) {
                    this.serverNameSet = true;
                    this.serverName = name;
                }
                return;
            }
            /**
             * Tomcat WEB Server
             */
            on = ObjectName.getInstance("Catalina:*");
            ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                this.domainName = "Catalina";
                this.serverName = this.targetName;
                this.serverNameSet = true;
                return;
            }
            // Jetty WEB Server
            on = ObjectName.getInstance("org.mortbay.jetty:type=server,*");
            ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                this.domainName = "org.mortbay.jetty";
                Iterator<?> onames = ons.iterator();
                ObjectName serverOn = (ObjectName) onames.next();
                String serverId = serverOn.getKeyProperty("id");
                if (serverId != null) {
                    serverName = serverId;
                } else {
                    serverName = this.targetName;
                }
                this.serverNameSet = true;
                return;
            }
        } catch (JMException e) {
            if (error == null) {
                error = e.toString();
                logger.warn("Cannot get server and domain names for target {0} ({1})", targetName, error);
            } else {
                if (!error.equals(e.toString())) {
                    // different error
                    error = e.toString();
                    logger.warn("Cannot get server and domain names for target {0} ({1})", targetName, error);
                }
            }
        } catch (IOException e) {
            if (error == null) {
                error = e.toString();
                logger.error("A communication problem occurred with target {0} ({1})", targetName, error);
            } else {
                if (!error.equals(e.toString())) {
                    // different error
                    error = e.toString();
                    logger.error("A communication problem occurred with target {0} ({1})", targetName, error);
                }
            }
        }

    }

    /**
     * Collect values of MBean attributes for this Collector.
     * @param cnx MBeanServerConnection
     * @param col JmxCollector
     * @return the Metrics for (all) the MBean(s) corresponding to the collector's
     * ObjectName pattern. All the MBeans correspond to one target ; so all the
     * Metrics have the same target.
     */
    private List<Metric> collect(MBeanServerConnection cnx, JmxCollector col) {
        LinkedList<Metric> metrics = new LinkedList<Metric>();
        try {
            ObjectName objname = col.getObjectName();
            if (objname.isPattern()) {
                logger.debug("collect pattern:" + objname.toString());
                // Must loop on all the mbeans matching the pattern
                Set<ObjectName> onset = JmxUtil.getMBeans(cnx, objname);
                if (onset.isEmpty()) {
                    logger.warn("No MBeans found for pattern {0} in this collector (indicator: {1})", objname.toString(), col.getIndicator().getName());
                }
                for (ObjectName on : onset) {
                    Metric m = getMBeanMetric(cnx, on, col);
                    if (m != null) {
                        metrics.add(m);
                    }
                }
            } else {
                logger.debug("collect single mbean:" + objname.toString());
                if (cnx.isRegistered(objname)) {
                    Metric m = getMBeanMetric(cnx, objname, col);
                    if (m != null) {
                        metrics.add(m);
                    }
                } else {
                    logger.error("Cannot collect from MBean {0} (does not exists)", objname);
                }
            }
        } catch (Exception e) {
            // TODO
            logger.warn("getAttributes error on JmxServer: " + e);
        }
        logger.debug("nb of metrics found: " + metrics.size());
        return metrics;
    }

    /**
     * Poll one MBean's attributes and get the values of the attributes and/or the fragments
     * corresponding to the indicator's definition.
     * If the indicator definition does not provides an attributes list, then all the attributes are polled.
     * @param cnx Connection to the MBean server
     * @param objname The MBean's ObjectName
     * @param col The current collector
     * @return A Metric object containing the attributes' and/or fragments' values,
     * or null if no attributes to poll (user needs to check indicator definition).
     * @throws IOException
     * @throws JMException
     */
    private Metric getMBeanMetric(MBeanServerConnection cnx, ObjectName objname, JmxCollector col) throws IOException, JMException {
        logger.debug("mbean:" + objname.toString());

        /**
         * All the attribute names in the MBean.
         */
        List<String> allAttNames = JmxUtil.getMBeanAttNames(cnx, objname);

        // The attributes to be used to construct the Metric
        AttributeList attl =  new AttributeList();

        long time = System.currentTimeMillis();

        if (!col.hasAttributes()) {
            // no names provided by the indicator, poll all the attributes in the MBean
            attl = getAllAttributes(cnx, objname, time, allAttNames);
        } else {
            attl = getAttributesForProvidedNames(cnx, objname, time, allAttNames, col.getName(), col.geAttrlist());
        }

        if (attl.isEmpty()) {
            logger.warn("No metrics found for indicator: " + col.getIndicator().getName()  + ", for mbean: " + objname.toString() + ", and target: " + targetName);
            return null;
        } else {
            Metric metric = new Metric(time, objname, attl, targetName);
            // add metadata
            metric.setProperty(JmxCollectorService.META_SERVER, serverName);
            metric.setProperty(JmxCollectorService.META_DOMAIN, domainName);
            metric.setProperty(JmxCollectorService.META_URL, url);
            return metric;
        }
    }

    /**
     * Get the all the attributes for this MBean.
     * @param cnx
     * @param objname
     * @param time
     * @param allAttNames
     * @return
     * @throws IOException
     * @throws JMException
     */
    private AttributeList getAllAttributes(final MBeanServerConnection cnx, final ObjectName objname, final long time, final List<String> allAttNames) throws IOException, JMException {
        AttributeList attl = cnx.getAttributes(objname, allAttNames.toArray(new String[0]));
        logger.debug("all attributes to poll:");
        for (String nameToPoll : allAttNames) {
            logger.debug(nameToPoll);
        }
        return attl;
    }

    /**
     * Get Attribute instances for all attribute names and/or attribute fragment names.
     * @param cnx
     * @param objname
     * @param time
     * @param allAttNames
     * @param colName
     * @param names
     * @return
     * @throws IOException
     * @throws JMException
     */
    private AttributeList getAttributesForProvidedNames(final MBeanServerConnection cnx, final ObjectName objname, final long time, final List<String> allAttNames, final String colName, final Collection<String> names) throws IOException, JMException {
        AttributeList attl =  new AttributeList();
        /**
         * map the provided names to MBean's attribute names based on the naming convention for attribute fragments
         */
        Map<String, String> attNamesToPoll = new HashMap<String, String>();
        /**
         * map the provided names to the MBean's Attributes.
         */
        Map<String, Attribute> attsToPoll = new HashMap<String, Attribute>();

        // parse the provided names and determine the name of the attributes to poll
        for (String name : names) {
            if (FragmentUtil.isFragmentName(name)) {
                try {
                    // fragment name
                    String attName = FragmentUtil.getAttributeName(name);
                    if (allAttNames.contains(attName)) {
                        attNamesToPoll.put(name, attName);
                    } else {
                        logger.error("Cannot poll fragment {0} (no attribute named {1} exists in MBean {2})", name, attName, objname);
                    }
                } catch (FragmentNameException e) {
                    logger.error("Cannot poll fragment {0}, incorrect format !");
                }
            } else {
                // attribute name
                if (allAttNames.contains(name)) {
                    attNamesToPoll.put(name, name);
                } else {
                    logger.error("Cannot poll attribute {0} (does not exists in MBean {1})", name, objname);
                }
            }
        }

        if (attNamesToPoll.keySet().isEmpty()) {
            logger.info("No attributes to poll in this MBean (based on collector {0} definition and {1} metadata)", colName, objname);
            return null;
        }

        logger.debug("names to poll:");
        for (String nameToPoll : attNamesToPoll.keySet()) {
            logger.debug(nameToPoll);
        }

        logger.debug("corresponding attribute names:");
        String[] attNames = new String[attNamesToPoll.keySet().size()];
        int i = 0;
        for (String nameToPoll : attNamesToPoll.keySet()) {
            attNames[i++] = attNamesToPoll.get(nameToPoll);
            logger.debug(attNamesToPoll.get(nameToPoll));
        }

        /**
         * The attributes to poll (corresponding to provided attribute names,
         * and/or the attribute name of the provided fragment names).
         */
        AttributeList atts = cnx.getAttributes(objname, attNames);

        for (String nameToPoll : attNamesToPoll.keySet()) {
            String attName = attNamesToPoll.get(nameToPoll);
            attsToPoll.put(nameToPoll, getAttribute(atts, attName));
        }

        /**
         * Construct <code>attl</code>.
         * For each Attribute in <code>atts</code>, insert it in <code>attl</code>, if its name is associated to an attribute name to poll.
         * Otherwise (its name is associated to an attribute fragment name), construct a new Attribute for that fragment and
         * insert it in <code>attl</code>.
         */
        for (String name : attsToPoll.keySet()) {
            if (FragmentUtil.isFragmentName(name)) {
                try {
                    Attribute att = JmxUtil.getAttributeFragment(name, attsToPoll.get(name));
                    attl.add(att);
                } catch (FragmentNameException e) {
                    logger.error("Cannot poll fragment {0} ({1})", name, e.toString());
                }
            } else {
                attl.add(attsToPoll.get(name));
            }

        }
        return attl;
    }

    /**
     * @param atts All the attributes to poll, contains the attribute to look for.
     * @param name Attribute name
     * @return the attribute corresponding to the given name.
     */
    private Attribute getAttribute(AttributeList atts, String attName) {
        for (Attribute att : atts.asList()) {
            if (att.getName().equals(attName)) {
                // found
                return att;
            }
        }
        // some problem occurred in the algorithm, normally we should have found an attribute
        return null;
    }

}
