/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.util.List;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.jmx.Metric;

/**
 * Collector implementation for JMX.
 * This implementation works with a separate thread used to pre-collect the results.
 * Using a thread is interesting when we do not want to be stuck in case info
 * cannot be got because the target is off.
 * @author durieuxp
 */
public class JmxCollector extends JmxCollectorBase {

    /**
     * Last result got by JmxWorker
     */
    private List<Metric> metrics = null;

    /**
     * Keeps last collected data during the construction of the JIVs.
     */
    private List<Metric> cache = null;

    /**
     * Set at true if the probe using this collector is stopped.
     */
    private boolean stopped = false;

    /**
     * Constructor
     * @param probeId probe Identifier
     * @param indicator JasmineIndicator definition
     * @param period in seconds
     * @param mbean MBean pattern
     * @param csv CSV list of attributes
     */
    public JmxCollector(String probeId, JasmineIndicator indicator, int period, String mbean, String csv) {
        super(probeId, indicator, period, mbean, csv);
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * This method return a List of results in case indicator represents
     * actually a list of values.
     * @return The constructed JasmineIndicatorValue
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        cache = metrics;
        metrics = null;
        // Create the JasmineIndicatorValue instance to return, based on data in cache
        if (cache == null) {
            logger.warn("Cannot return value for indicator {0} (no available metrics)", getIndicatorName());
            return null;
        }
        JasmineIndicatorValue jiv = getJiv(getIndicatorName(), getProbeId(), getIndicator().getScale(), cache);
        return jiv;
    }

    /**
     * Stop polling
     */
    @Override
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        stopped = true;
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        stopped = false;
    }

    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    /**
     * Check if this Collector needs a new result.
     * A stopped collector returns false.
     * Otherwise, it returns true if there are nor metrics, or if metrics are out-dated
     * regarding to the poll period.
     *
     * @return true if it needs one
     */
    public boolean needResult() {
        if (stopped) {
            logger.debug("STOPPED, need no result for " + indicator.getName() + " in probe " + probeId);
            return false;
        }
        if (metrics == null) {
            return true;
        }
        long timestamp = metrics.get(0).getTimestamp();
        if (oldvalue(timestamp)) {
            // TODO ? to check with Ph.
            // forget this value too old.
            // metrics = null;
            return true;
        }
        return false;
    }

    /**
     * Add a new result to the list.
     * @param metrics
     */
    public void addResult(List<Metric> metrics) {
        if (metrics.size() == 0) {
            logger.debug("Empty metric list");
            return;
        }
        logger.debug("");
        this.metrics = metrics;
    }

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * @return true if this timestamp is too old for the period.
     */
    private boolean oldvalue(long timestamp) {
        long now = System.currentTimeMillis();
        return (now - timestamp > period * 1000);
    }

}
