/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Manage JMX Attribute fragments.
 *
 * Attribute fragments are elements of complex attributes.
 *
 * Sometimes, users need to poll such attribute elements, for example an item in a Map,
 * and don't need the entire attribute value.
 * A naming policy is necessary to define the attributes elements to be polled by a JMX indicator.
 *
 * Also, when a JMX indicator specifies an attribute (or an attribute fragment) whose value has a complex
 * type, the value must be decomposed in simple elements (elementary fragments).
 * The naming policy is used to identify the fragments returned by a probe polling this indicator.
 *
 * @author danesa
 */
public class FragmentUtil {

    private static Log logger = LogFactory.getLog(JmxCollector.class);


    // =======================
    //
    // Naming policy
    //
    // =======================
    public static String DOT = ".";
    public static String COMMA = ",";
    public static String BRACKLEFT = "[";
    public static String BRACKRIGHT= "]";

    /**
     *
     * @param name attribute or fragment name
     * @return true if the name has a fragment name pattern
     */
    public static boolean isFragmentName(String name) {
        if (name.contains(DOT)) {
            return true;
        } else if (name.contains(BRACKLEFT) && name.contains(BRACKLEFT) && name.indexOf(BRACKLEFT) < name.indexOf(BRACKRIGHT)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Generate the name of a fragment corresponding to a CompositeData's item.
     * @param name composite's name
     * @param key the item's name
     */
    public static String getNameForItem(String name, String key) {
        String itemName = name + DOT + key;
        return itemName;
    }

    /**
     * Generate the name of a fragment corresponding to an Array or TabularData's item
     * @param name
     * @param elemName
     * @return
     */
    public static String getNameForElem(String name, String elemName) {
        // elemName format : elemBRACKLEFTindexBRACKRIGHT
        String elem = elemName.substring(0, elemName.indexOf(BRACKLEFT));
        // TODO get the index of the last occurrence of elem !!!!
        // (for the moment suppose only one occurrence exists !!!
        int elemIndex = name.indexOf(elem);
        String name_firstPart = name.substring(0, elemIndex);
        String name_secondPart = elemName;
        return new String(name_firstPart + name_secondPart);
    }

    /**
     * Generate the name of an element of an Array.
     * @param name the name of the Array attribute or fragment
     * @param index the element's index
     * @return
     */
    public static String getNameForArrayElement(String name, String index) {
        String elemName = name + BRACKLEFT + index + BRACKRIGHT;
        return elemName;
    }

    /**
     * Generate the name of a fragment corresponding to a row of in a TabularData
     * @param name the name of the TabularData attribute or fragment
     * @param indexes the row's indexes
     * @return
     */
    public static String getNameForTabularElement(String name, Object[] indexes) {
        String s_indexes = indexes[0].toString();
        if (indexes.length > 1) {
            for (int i = 1; i < indexes.length; i++) {
                s_indexes = s_indexes + COMMA + indexes[i];
            }
        }
        String elemName = name + BRACKLEFT + s_indexes + BRACKRIGHT;
        return elemName;
    }


    /**
     * Get the name of an item's key from a fragment name.
     * Suppose that the fragment name has the following format: nameDOTkey
     * @param fragmentName fragment name
     * @return key
     * @throws FragmentNameException A key name could not be identified.
     */
    public static String getItemName(String fragmentName) throws FragmentNameException {
        int dotIndex = fragmentName.indexOf(DOT);
        if (dotIndex > 0) {
            return fragmentName.substring(dotIndex + 1);
        } else {
            throw new FragmentNameException("Can't get item name from: " + fragmentName);
        }
    }

    /**
     * Get the name of an element from a fragment name.
     * Suppose that the fragment name has the following format: nameBRACKLEFTindexesBRACKRIGHT
     * @param fragmentName fragment name
     * @return the element name
     */
    public static String getElementName(String fragmentName) throws FragmentNameException {
        int brackLeftIndex = fragmentName.indexOf(BRACKLEFT);
        int brackRightIndex = fragmentName.indexOf(BRACKRIGHT);
        if (brackLeftIndex > 0 && brackRightIndex > 0 && brackLeftIndex < brackRightIndex) {
            return fragmentName;
        } else {
            throw new FragmentNameException("Can't get element name from: " + fragmentName);
        }
    }

    //////////////////////////////////////////////////////////////////////////////////////////////


    /**
     * Generate the name of an element of indexed object.
     * @param name the name associated to the indexed object.
     * @param index the element's index
     * @return
     */
    public static String nameForIndexElement(String name, String index) {
        String elemName = name + BRACKLEFT + index + BRACKRIGHT;
        return elemName;
    }

    /**
     * Generate the name of a fragment corresponding to a CompositeData's item.
     * @param name composite's name
     * @param key the item's name
     */
    public static String nameForItem(String name, String key) {
        String itemName = name + DOT + key;
        return itemName;
    }

    /**
     * Get the attribute name from the attribute fragment name
     * @param name attribute fragment name
     * @return attribute name
     * @throws FragmentNameException An attribute name could not be identified as no
     * DOT nor BRACKLEFT found.
     */
    public static String getAttributeName(final String name) throws FragmentNameException {
        int index; // index of the first attribute element
        int dotIndex = name.indexOf(DOT);
        int brackLeftIndex = name.indexOf(BRACKLEFT);
        if (dotIndex < 0 && brackLeftIndex <0) {
            throw new FragmentNameException("Can't get attribute name from: " + name);
        }
        if (dotIndex > 0 && brackLeftIndex < 0) {
            index = dotIndex;
        } else if (brackLeftIndex > 0 && dotIndex < 0) {
            index = brackLeftIndex;
        } else {
            if (dotIndex < brackLeftIndex) {
                index = dotIndex;
            } else {
                index = brackLeftIndex;
            }
        }
        return name.substring(0, index);
    }

    public static String getKeyName(final String name, final String attributeName) {
        // no check of attributeName
        // suppose name has form <attributeName><elements>
        // where element has form <key>..., where <key> has form <DOT><key name>
        String keyName;
        String elements = name.substring(attributeName.length());
        int dotIndex = elements.indexOf(DOT);
        elements = elements.substring(dotIndex +1);
        // check for subsequent DOT or BRACKLEFT
        dotIndex = elements.indexOf(DOT);
        int brackLeftIndex = elements.indexOf(BRACKLEFT);
        if (dotIndex < 0 && brackLeftIndex <0) {
            return elements;
        }
        if (dotIndex > 0 && brackLeftIndex < 0) {
            return elements.substring(0, dotIndex);
        } else if (brackLeftIndex > 0 && dotIndex < 0) {
            return elements.substring(0, brackLeftIndex);
        } else {
            if (dotIndex < brackLeftIndex) {
                return elements.substring(0, dotIndex);
            } else {
                return elements.substring(0, brackLeftIndex);
            }
        }
    }

    /**
     * Get the indexes in an indexed attribute element having the following expected format:
     * <attributeName><indexExpr>..., where <indexExpr> has form <BRACKLEFT>indexes<BRACKRIGHT>
     * @param name The String to process.
     * @param attributeName The supposed starting attribute name
     * @return the found indexes
     * @throws FragmentNameException the given String does not have the expected format
     */
    public static String getIndexes(final String name, final String attributeName) throws FragmentNameException {
        // no check of attributeName
        // suppose name has form <attributeName><indexExpr>..., where <indexExpr> has form <BRACKLEFT>indexes<BRACKRIGHT>
        try {
        int brackLeftIndex = name.indexOf(BRACKLEFT);
        int brackRightIndex = name.indexOf(BRACKRIGHT);
        String indexes = name.substring(brackLeftIndex + 1, brackRightIndex);
        return indexes;
        } catch (IndexOutOfBoundsException e) {
            throw new FragmentNameException("Malformed fragment name: " + name + ". It was expected to start with " + attributeName + " and continue with an indexed element");

        }
    }

    /**
     * Get the first attribute element's name in a String having the following expected format:
     * <attributeName><attributeElementName>...
     * @param name The String to process.
     * @param attributeName The supposed starting attribute name
     * @return the element name
     * @throws FragmentNameException
     */
    public static String getElementName(final String name, final String attributeName) throws FragmentNameException {
        // check format
        // suppose name has form <attributeName><attribute element>...
        if (name.indexOf(attributeName) != 0) {
            // name does not starts with attributeName
            throw new FragmentNameException("Malformed fragment name: " + name + ". It was expected to start with " + attributeName);
        }
        try {
            String elements = name.substring(attributeName.length(), name.length());
            if (elements.startsWith(DOT)) {
                return getKeyName(name, attributeName);
            }
            if (elements.startsWith(BRACKLEFT)) {
                if (elements.indexOf(BRACKRIGHT) > 0) {
                    return getIndexes(name, attributeName);
                } else {
                    // wrong format, starts with BRACKLEFT but no BRACKRIGHT after it
                    throw new FragmentNameException("Malformed fragment name: " + ". Didn't found expected right bracket.");
                }
            }
            // didn't recognize attribute element
            throw new FragmentNameException("Malformed fragment name: " + name + ". Didn't recognize any attibute element after " + attributeName);

        } catch (IndexOutOfBoundsException e) {
            // attributeName is not followed by elements
            throw new FragmentNameException("Malformed fragment name: " + name + ". It was expected to start with " + attributeName);

        }
    }

    /**
     * Check if the first attribute element in the given String corresponds to a key element
     * @param name A String having the expected format: <attributeName><attributeElementName>...
     * @param attributeName The supposed starting attribute name
     * @return true if the first attribute element has the expected format: .<keyName>
     * @throws FragmentNameException
     */
    public static boolean nextElementIsKey(final String name, final String attributeName) throws FragmentNameException {
        // check format
        // suppose name has form <attributeName><attribute element>...
        if (name.indexOf(attributeName) != 0) {
            // name does not starts with attributeName
            throw new FragmentNameException("Malformed fragment name: " + name + ". It was expected to start with " + attributeName);
        }
        try {
            String elements = name.substring(attributeName.length(), name.length());
            if (elements.startsWith(DOT)) {
                return true;
            } else {
                return false;
            }
        } catch (IndexOutOfBoundsException e) {
            // attributeName is not followed by elements
            throw new FragmentNameException("Malformed fragment name: " + name + ". An attribute element expected after " + attributeName);

        }
    }

    /**
     * Check if the first attribute element in the given String corresponds to an indexed element.
     * @param name A String having the expected format: <attributeName><attributeElementName>...
     * @param attributeName The supposed starting attribute name
     * @return true if the first attribute element has the expected format: [index]
     * @throws FragmentNameException
     */
    public static boolean nextElementIsIndex(final String name, final String attributeName) throws FragmentNameException {
        // check format
        // suppose name has form <attributeName><attribute element>...
        if (name.indexOf(attributeName) != 0) {
            // name does not starts with attributeName
            throw new FragmentNameException("Malformed fragment name: " + name + ". It was expected to start with " + attributeName);
        }
        try {
            String elements = name.substring(attributeName.length(), name.length());
            if (elements.startsWith(BRACKLEFT)) {
                if (elements.indexOf(BRACKRIGHT) > 0) {
                    return true;
                } else {
                    // wrong format, starts with BRACKLEFT but no BRACKRIGHT after it
                    throw new FragmentNameException("Malformed fragment name: " + ". Didn't found expected right bracket.");
                }
            }
            return false;
        } catch (IndexOutOfBoundsException e) {
            // attributeName is not followed by elements
            throw new FragmentNameException("Malformed fragment name: " + name + ". An attribute element expected after " + attributeName);
        }
    }
}
